﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  デーモンの動作を制御するインターフェースです。
 */

#include <nn/nn_ApplicationId.h>
#include <nn/account/account_Types.h>
#include <nn/olsc/olsc_ForbiddenSaveDataIndication.h>
#include <nn/olsc/olsc_Stopper.h>

namespace nn { namespace olsc {
    namespace srv {
        class IDaemonController;
    }

    /**
    *       @brief      デーモンの制御を行うクラスです。
    *       @details    バックグラウンドで稼働するデーモンの動作を制御することができます。
    *                   デーモンはバックグラウンドで下記の動作を行います。
    *
    *                   - データの転送
    *                   - データ情報キャッシュ更新
    *                   - ポリシーキャッシュ更新
    *
    *                   これらの動作によって、UI 側で参照するデータに対して不整合が発生する場合があります。
    *                   必要に応じて Stop 関数を呼び出して、デーモンの動作を抑制するようにしてください。
    *
    */
    class DaemonController
    {
        NN_DISALLOW_COPY(DaemonController);

    public:
        /**
        *       @brief                      コンストラクタです。デーモンの動作を制御するインスタンスを作成します。
        */
        DaemonController() NN_NOEXCEPT;
        ~DaemonController() NN_NOEXCEPT;

        NN_IMPLICIT DaemonController(DaemonController&& rhs) NN_NOEXCEPT;
        DaemonController& operator=(DaemonController&& rhs) NN_NOEXCEPT;


        /**
        *       @brief      指定したアプリケーションに対して、セーブデータの自動アップロードが有効か無効かを取得します。
        *
        *       @param[in]  uid         対象のユーザ。
        *       @param[in]  appId       対象のアプリケーション。
        *
        *       @return     true なら自動アップロードが有効になっている。
        *
        */
        bool GetApplicationAutoUploadSetting(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT;

        /**
        *       @brief      指定したアプリケーションに対して、セーブデータの自動アップロードの有効無効を設定します。
        *
        *       @param[in]  uid         対象のユーザ。
        *       @param[in]  appId       対象のアプリケーション。
        *       @param[in]  enabled     true なら自動アップロードを有効にする。
        *
        *       @details    true に設定した場合、そのタイミングでアップロードの転送タスクが登録されます。
        *
        */
        void SetApplicationAutoUploadSetting(const account::Uid& uid, ApplicationId appId, bool enabled) NN_NOEXCEPT;


        /**
        *       @brief      指定したアカウントのセーブデータの自動アップロードの有効無効を設定します。
        *
        *       @param[in]  uid         対象のユーザ。
        *       @param[in]  enabled     true なら自動アップロードを有効にする。
        *
        *       @details    true の状態から false にした場合、その時点で積まれている自動アップロードタスクは全て削除されます。
        *                   true を設定するまで、そのユーザに対するすべての自動アップロードは発生しません。
        *                   true に設定した場合、適当なタイミングで自動アップロードタスクが積まれます。
        *                   必ずしも、true にした直後に登録されないことに注意してください。
        *
        */
        void SetGlobalAutoUploadSetting(const account::Uid& uid, bool enabled) NN_NOEXCEPT;


        /**
        *       @brief      指定したアカウントのセーブデータの自動アップロードの有効無効の設定を取得します。
        *
        *       @param[in]  uid         対象のユーザ。
        *       @return     true なら自動アップロードが有効。
        *
        */
        bool GetGlobalAutoUploadSetting(const account::Uid& uid) const NN_NOEXCEPT;


        /**
        *       @brief      デーモンによる自律タスクの実行を停止します。
        *
        *       @pre        StopAutonomyTaskExecution が呼び出されていない。
        *
        *       @return     自律動作の停止状態を管理するオブジェクトを返します。
        *
        *       @details    この関数が呼ばれると、デーモンの自律的なタスクの実行を停止します。
        *                   関数呼び出し時に返されるオブジェクトが破棄されると停止状態が解除されます。
        *
        *                   RemoteStorageController::ListDataInfo や、TransferTaskListController::List を複数回呼び出すようなケースでは
        *                   デーモンの動作に依って整合性が崩れるのを防ぐために、あらかじめこの関数を呼ぶようにしてください。
        *
        *                   また本体機能によるセーブデータ削除や、ユーザ移行の際にもこの関数を呼び意図しないデーモンの自律動作を防ぐようにしてください。
        */
        Stopper StopAutonomyTaskExecution() NN_NOEXCEPT;


        /**
        *       @brief  将来削除されます。代わりに SetApplicationAutoUploadSetting を使用してください。
        */
        void SetAutoUploadEnabled(const account::Uid& uid, ApplicationId appId, bool enabled) NN_NOEXCEPT;

        /**
        *       @brief  将来削除されます。代わりに GetApplicationAutoUploadSetting を使用してください。
        */
        bool IsAutoUploadEnabled(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT;

        // 内部利用
        ForbiddenSaveDataIndication CreateForbiddenSaveDataInidication(ncm::ApplicationId appId) NN_NOEXCEPT;

    private:
        friend DaemonController OpenDaemonController() NN_NOEXCEPT;
        friend void swap(DaemonController& lhs, DaemonController& rhs) NN_NOEXCEPT;

        NN_IMPLICIT DaemonController(srv::IDaemonController* interface) NN_NOEXCEPT;

        srv::IDaemonController* m_Interface;
    };
}} // namespace nn::olsc
