﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   テレビとの電源連動に関する機能
 *
 */

#pragma once

#include <nn/nn_Macro.h>

namespace nn { namespace oe {

/**
   @brief   テレビとの電源連動機能の設定値を切り替えるための列挙型です。

 */
enum TvPowerStateMatchingMode
{
    TvPowerStateMatchingMode_Default  = 0,  //!< 本体設定に従います
    TvPowerStateMatchingMode_Disabled = 1,  //!< テレビとの電源連動を無効にします
};

//--------------------------------------------------------------------------

//! @name   テレビとの電源連動機能
//! @{

/**
    @brief  テレビとの電源連動機能の設定を変更します

    @param[in] mode  テレビとの電源連動モードを示す列挙子

    @details
     HDMI-CEC 制御による Switch 本体とテレビとの電源連動機能を、
     mode で指定された設定内容に変更します。

     mode に nn::oe::TvPowerStateMatchingMode_Disabled を指定した場合、
     以下のような機能が無効化されます。

     - Switch 本体をドックインした時にテレビの電源を自動的に ON にする
     - テレビの電源 OFF に伴って Switch 本体を自動的にスリープさせる

     この設定は自プログラムがインフォーカス状態の時にのみ効果があります。
     自身がインフォーカス状態でなくなった際には、実質的に
     nn::oe::TvPowerStateMatchingMode_Default 相当となって本体設定値に
     従った状態になります。その後再びインフォーカス状態に戻ったら、
     この関数で設定した値が再度有効となります。
     なお、プログラム起動時は nn::oe::TvPowerStateMatchingMode_Default
     となっています。

     また、本 API を発行しても本体設定の設定値は変更されません。

*/
void SetTvPowerStateMatchingMode(TvPowerStateMatchingMode mode) NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::oe

