﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   無操作状態の制御に関する API の宣言
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>

namespace nn { namespace oe {

//--------------------------------------------------------------------------

/**
 * @brief   自動スリープや画面焼け軽減の開始時間の延長モードです。
 */
enum UserInactivityDetectionTimeExtensionMode
{
    UserInactivityDetectionTimeExtensionMode_Disabled = 0,      //!< 延長モードが無効
    UserInactivityDetectionTimeExtensionMode_Enabled = 1,       //!< 延長モードが有効
    UserInactivityDetectionTimeExtensionMode_EnabledUnsafe = 2, //!< 長時間の安全でない延長モードが有効
};

//! @name   無操作状態の制御に関する API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態を検出した際に働く自動スリープや画面焼け軽減の開始時間を延長します。
 *
 * @param[in] isExtended    延長機能の有効・無効を bool 型で指定します。
 *
 * @details
 *  無操作状態を検出した際に働く自動スリープや画面焼け軽減の開始時間を延長する機能を有効もしくは無効にします。
 *  isExtended が true の場合は有効となります。
 *  isExtended が false の場合は無効となり、通常のポリシーによる制御に戻します。
 *  @n
 *  本関数は、スタッフロール中など、ユーザがしばらく無操作の状態になることを予期したシーンにおいて利用されることを想定しています。 @n
 *  @n
 *  本関数により延長を有効にすると、通常本体設定や動作モード等に基づいて決定される
 *  自動スリープや画面焼け軽減の開始時間ポリシーがそれぞれ以下のように延長されます。
 *
 *  @li 画面焼け軽減の開始条件となる無操作状態の継続時間が 15 分未満であったときに、これを 15 分まで延長する
 *  @li 自動スリープの開始条件となる無操作状態の継続時間が 30 分未満であったときに、これを 30 分まで延長する
 *
 *  例えば、本体設定がユーザによって以下のように設定されていたとします。
 *
 *  | 本体設定の項目名                     | 設定値 |
 *  | ------------------------------------ | ------ |
 *  | 本体画面であそぶときの自動スリープ   | 1 分   |
 *  | テレビであそぶときの自動スリープ     | 1 時間 |
 *  | コンテンツ再生中は自動スリープしない | ON     |
 *
 *  このとき、通常ポリシーでは、無操作状態での自動スリープと画面焼け軽減の開始時間は下表のようになります。
 *
 *  | 動作モード          | 画面焼け軽減開始時間 | 自動スリープ開始時間 |
 *  | ------------------- | -------------------- | -------------------- |
 *  | 本体画面表示中      | 45 秒                | 1 分                 |
 *  | テレビ表示中        | 5 分                 | 1 時間               |
 *
 *  本関数を呼び出して延長機能を有効にすると、これらの条件は下表のように変化します。
 *
 *  | 動作モード          | 画面焼け軽減開始時間 | 自動スリープ開始時間 |
 *  | ------------------- | -------------------- | -------------------- |
 *  | 本体画面表示中      | 15 分                | 30 分                |
 *  | テレビ表示中        | 15 分                | 1 時間               |
 *
 *  本関数を呼び出すことで無操作状態の継続時間のカウントがリセットされることはありません。@n
 *  たとえば、無操作状態が 4 分継続した時点で本関数を呼び出した場合、そのまま無操作状態が続けば、
 *  本関数を呼び出して 11 分後に画面焼け軽減が開始します。 @n
 *  @n
 *  また、本関数を呼び出すことですでに発動した各機能が解除されることもありません。 @n
 *  たとえば、無操作状態が 5 分継続して画面焼け軽減が開始してから、その 1 分後に本関数を呼び出しても、
 *  すでに発動した画面焼け軽減が延長によって解除されるわけではありません。 @n
 *  この場合は、ユーザが操作するなどして一度無操作状態が解除されてから、延長したポリシーが適用されます。 @n
 *  この場合でも自動スリープはまだ発動していませんので、こちらは延長したポリシーがすぐに適用されます。 @n
 *  @n
 *  本機能の効果は、本関数を呼び出したアプリケーションにフォーカスがある間のみ有効です。 @n
 *  @n
 *  本関数による延長と nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe() による延長は排他です。 @n
 *  本関数を false で呼び出した場合はどちらの延長機能も無効となります。 @n
 *
 * @see
 *  nn::oe::IsUserInactivityDetectionTimeExtended()
 *  nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe()
 *
 */
void SetUserInactivityDetectionTimeExtended(bool isExtended) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態を検出した際に働く自動スリープや画面焼け軽減の開始時間を、より長時間にわたり延長します。
 *
 * @param[in] isExtended    延長機能の有効・無効を bool 型で指定します。
 *
 * @details
 *  無操作状態を検出した際に働く自動スリープや画面焼け軽減の開始時間を、より長時間にわたり延長する機能を有効もしくは無効にします。 @n
 *  @n
 *  基本的な挙動は nn::oe::SetUserInactivityDetectionTimeExtended() と同じですが、より長時間延長できる一方、
 *  この機能を使用すると、画面保護のため新たに遵守が必要となるガイドライン項目が発生することに注意してください。 @n
 *  @n
 *  本関数により延長を有効にすると、通常本体設定や動作モード等に基づいて決定される
 *  自動スリープや画面焼け軽減の開始時間ポリシーがそれぞれ以下のように延長されます。
 *
 *  @li 画面焼け軽減は無効となる
 *  @li 自動スリープの開始条件となる無操作状態の継続時間が 1 時間未満であったときに、これを 1 時間まで延長する
 *
 *  例えば、本体設定がユーザによって以下のように設定されていたとします。
 *
 *  | 本体設定の項目名                     | 設定値 |
 *  | ------------------------------------ | ------ |
 *  | 本体画面であそぶときの自動スリープ   | 1 分   |
 *  | テレビであそぶときの自動スリープ     | 1 時間 |
 *  | コンテンツ再生中は自動スリープしない | ON     |
 *
 *  このとき、通常ポリシーでは、無操作状態での自動スリープと画面焼け軽減の開始時間は下表のようになります。
 *
 *  | 動作モード          | 画面焼け軽減開始時間 | 自動スリープ開始時間 |
 *  | ------------------- | -------------------- | -------------------- |
 *  | 本体画面表示中      | 45 秒                | 1 分                 |
 *  | テレビ表示中        | 5 分                 | 1 時間               |
 *
 *  本関数を呼び出して延長機能を有効にすると、これらの条件は下表のように変化します。
 *
 *  | 動作モード          | 画面焼け軽減開始時間 | 自動スリープ開始時間 |
 *  | ------------------- | -------------------- | -------------------- |
 *  | 本体画面表示中      | 無効                 | 1 時間               |
 *  | テレビ表示中        | 無効                 | 1 時間               |
 *
 *  本関数を呼び出すことで無操作状態の継続時間のカウントがリセットされることはありません。@n
 *  たとえば、無操作状態が 4 分継続した時点で本関数を呼び出した場合、そのまま無操作状態が続けば、
 *  本関数を呼び出して 56 分後に自動スリープが開始します。 @n
 *  @n
 *  また、本関数を呼び出すことですでに発動した各機能が解除されることもありません。 @n
 *  たとえば、無操作状態が 5 分継続して画面焼け軽減が開始してから、その 1 分後に本関数を呼び出しても、
 *  すでに発動した画面焼け軽減が延長によって解除されるわけではありません。 @n
 *  この場合は、ユーザが操作するなどして一度無操作状態が解除されてから、延長したポリシーが適用されます。 @n
 *  この場合でも自動スリープはまだ発動していませんので、こちらは延長したポリシーがすぐに適用されます。 @n
 *  @n
 *  本機能の効果は、本関数を呼び出したアプリケーションにフォーカスがある間のみ有効です。 @n
 *  @n
 *  本関数による延長と nn::oe::SetUserInactivityDetectionTimeExtended() による延長は排他です。 @n
 *  本関数を false で呼び出した場合はどちらの延長機能も無効となります。 @n
 *
 * @see
 *  nn::oe::IsUserInactivityDetectionTimeExtended()
 *  nn::oe::SetUserInactivityDetectionTimeExtended()
 *
 */
void SetUserInactivityDetectionTimeExtendedUnsafe(bool isExtended) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   無操作状態を検出した際に働く自動スリープや画面焼け軽減の開始時間の延長機能が有効であるかを取得します。
 *
 * @return  延長機能が現在有効であるかどうかが返ります。
 *
 * @details
 *  無操作状態を検出した際に働く自動スリープや画面焼け軽減機能の開始時間の延長機能が現在有効であるかを取得します。 @n
 *  @n
 *  nn::oe::SetUserInactivityDetectionTimeExtended() または nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe() を
 *  true を指定して呼び出した場合に、本関数は true を返します。
 *
 * @see
 *  nn::oe::SetUserInactivityDetectionTimeExtended()
 *  nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe()
 *  nn::oe::GetUserInactivityDetectionTimeExtensionMode()
 *
 */
bool IsUserInactivityDetectionTimeExtended() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   無操作状態を検出した際に働く自動スリープや画面焼け軽減の開始時間の現在の延長モードを取得します。
 *
 * @return  現在の延長モードが返ります。
 *
 * @details
 *  無操作状態を検出した際に働く自動スリープや画面焼け軽減機能の開始時間の現在の延長モードを取得します。 @n
 *
 * @see
 *  nn::oe::IsUserInactivityDetectionTimeExtended()
 *
 */
UserInactivityDetectionTimeExtensionMode GetUserInactivityDetectionTimeExtensionMode() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::oe

