﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/time/time_PosixTime.h>
#include <nn/util/util_TypedStorage.h>

#include <nn/ntc/ntc_EnsureNetworkClockAvailabilityMode.h>

namespace nn { namespace ntc { namespace shim {

    /**
     * @brief   ネットワーク時計の有効性を確立します。
     * @return  処理結果を返します
     * @retval  ResultSuccess   成功しました
     * @retval  上記以外        失敗しました
     * @pre
     *  - ネットワーク接続管理ライブラリ (NIFM) でネットワーク利用要求が受理されている
     * @details
     *  本関数は処理終了までブロックします。
     *
     *  Windows 版で本関数を利用するには、事前に以下の準備が必要です。
     *  - Socket ライブラリが初期化済である
     *  - libcurl ライブラリが初期化済である
     *  - 時刻ライブラリが初期化済である
     */
    nn::Result EnsureNetworkClockAvailability() NN_NOEXCEPT;

     /**
     * @brief   ネットワーク時計の有効性の確立処理をキャンセルします。
     *
     * @details
     *  @ref EnsureNetworkClockAvailability をキャンセルする場合、
     *  別スレッドから本関数を利用してください。
     */
    void CancelEnsuringNetworkClockAvailability() NN_NOEXCEPT;

    /**
     * @brief   バックグラウンドで行われる時刻の自動補正処理を抑止します。
     * @details
     *  自身のプロセスに限らず、一度でも本関数が呼ばれると自動補正処理は停止されます。
     *  本体システムのシャットダウンによって設定は無効化されます。
     *
     *  本関数を呼んだのと同じ回数 ResumeAutonomicTimeCorrection() を呼ぶことで、自動補正処理が再開されます。
     *
     *  Windows 実行環境では、事前に NIFM ライブラリが初期化済である必要があります。
     */
    void SuspendAutonomicTimeCorrection() NN_NOEXCEPT;

    /**
     * @brief   バックグラウンドで行われる時刻の自動補正処理を再開します。
     * @pre
     *  - 自動補正処理が停止状態である
     * details
     *  SuspendAutonomicTimeCorrection() を呼んだ回数と同じ回数分本関数を呼ぶことで、自動補正処理が再開されます。
     */
    void ResumeAutonomicTimeCorrection() NN_NOEXCEPT;

    /**
     * @brief   ネットワーク時計の有効性確立を非同期に行うためのクラスです
     * @details
     *  CorrectionNetworkClockAsyncTask クラスのオブジェクトを大域的に配置することはできません。
     */
    class CorrectionNetworkClockAsyncTask
    {
    public:
        /**
         * @brief   コンストラクタです。
         * @param[in]   eventClearMode                      GetFinishNotificationEvent() で得られるシステムイベントのクリアモードを指定します。
         * @param[in]   ensureNetworkClockAvailabilityMode  有効性確立のモードを指定します。
         * @details
         *  このコンストラクタの実行前にネットワーク接続ライブラリを初期化する必要があるため、
         *  CorrectionNetworkClockAsyncTask クラスのオブジェクトを大域的に配置することはできません。
         */
        CorrectionNetworkClockAsyncTask(
            nn::os::EventClearMode eventClearMode,
            nn::ntc::EnsureNetworkClockAvailabilityMode ensureNetworkClockAvailabilityMode) NN_NOEXCEPT;

        ~CorrectionNetworkClockAsyncTask() NN_NOEXCEPT;

        /**
         * @brief   ネットワーク時計の有効性確立の非同期処理を開始します。
         * @return  処理結果を返します
         * @retval  ResultSuccess   成功しました
         * @retval  上記以外        失敗しました
         * @details
         *  StartTask() を実行後に GetFinishNotificationEvent() で得られる
         *  システムイベントのシグナルを検知することで、非同期処理の完了を知ることができます。
         *
         *  既に処理中である場合に再度 StartTask() を実行しても、処理は一度しか行われません。
         *
         *  GetFinishNotificationEvent() で得られるシステムイベントがシグナル後であれば、
         *  再度 StartTask() で処理を再実行することができます。
         */
        nn::Result StartTask() NN_NOEXCEPT;

        /**
         * @brief   非同期処理の結果を取得します
         * @return  処理結果を返します
         * @retval  ResultSuccess   成功しました
         * @retval  上記以外        失敗しました
         * @details
         *  GetFinishNotificationEvent() で得られるシステムイベントがシグナル後に、
         *  処理の正しい結果が得られるようになります。
         *
         *  StartTask() 実行前や、システムイベントのシグナル前に呼び出しても副作用はありません。
         */
        nn::Result GetResult() NN_NOEXCEPT;

        /**
         * @brief   非同期処理の完了時にシグナルされるシステムイベントを取得します。
         * @details
         *  StartTask() 実行前に呼び出しても副作用はありません。@n
         *  その時のシステムイベントは非シグナル状態です。
         */
        nn::os::SystemEvent& GetFinishNotificationEvent() NN_NOEXCEPT;

        /**
         * @brief   実行中の非同期処理をキャンセルします。
         * @details
         *  StartTask() 実行前に呼び出しても副作用はありません。
         *
         *  キャンセルが正常に受け付けられた場合、
         *  GetFinishNotificationEvent() で得られるシステムイベントがシグナル状態に遷移し、
         *  GetResult() はキャンセルされたことを示す値を返すようになります。
         */
        void Cancel() NN_NOEXCEPT;

        /**
         * @breif   非同期処理が実行中かどうかを返します。
         * @return  非同期処理が実行中かどうか
         * @retval  true    非同期処理が実行中である
         * @retval  false   非同期処理が実行中でない
         */
        bool IsProcessing() NN_NOEXCEPT;

        /**
         * @brief   サーバー時刻を取得します。(EnsureNetworkClockAvailabilityMode_GetServerTime モード専用)
         * @return  サーバーからダウンロードした時刻を返します。
         * @pre
         *  - EnsureNetworkClockAvailabilityMode_GetServerTime モードがコンストラクタで指定されている
         *  - タスク処理が一度でも成功している
         * @details
         *  タスク成功後に時間が経過しても、取得できる時刻は変わりません。
         *  最新の時刻を知りたい場合はタスクを再実行してください。
         */
        nn::time::PosixTime GetServerPosixTime() NN_NOEXCEPT;

        typedef nn::util::TypedStorage<char, 128u, 8u> StorageType;

    private:
        StorageType m_Storage;
    };

}}} // nn::ntc::shim
