﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ntc/detail/service/ntc_Common.h>

namespace nn { namespace ntc { namespace detail { namespace service {

/**
 * @brief   自律的にネットワーク時計の補正処理を有効性確立を行うスレッドクラス
 */
class AutonomicEnsureNetworkClockAvailabilityThread
{
    NN_DISALLOW_COPY(AutonomicEnsureNetworkClockAvailabilityThread);
    NN_DISALLOW_MOVE(AutonomicEnsureNetworkClockAvailabilityThread);

public:
    AutonomicEnsureNetworkClockAvailabilityThread(
        nn::TimeSpan intervalTimeSpan,
        nn::TimeSpan retryTimeSpan,
        int immediateTryCountMax,
        nn::TimeSpan immediateTryTimeSpan) NN_NOEXCEPT;

    //!< スレッド開始
    void StartThread(char* stack, size_t stackSize, int priority, const char* threadName) NN_NOEXCEPT;

    //!< スレッド終了
    void StopThread() NN_NOEXCEPT;

    //!< 補正処理を即座に試行する
    void ImmediatelyExecute() NN_NOEXCEPT
    {
        NN_DETAIL_NTC_SERVER_LOG("AutonomicEnsureNetworkClockAvailabilityThread ImmediateExecutionEvent signal.\n");
        m_ImmediateExecutionEvent.Signal();
    }

    //!< 補正処理の停止
    Result Suspend() NN_NOEXCEPT;

    //!< 補正処理の再開
    Result Resume() NN_NOEXCEPT;

private:
    nn::os::ThreadType m_Thread;
    nn::os::Event m_StopEvent; //!< スレッド終了要求があればシグナルするイベント
    nn::os::Event m_ImmediateExecutionEvent; //!< 補正を即座に行う必要があればシグナルするイベント

    nn::os::Event m_ResumeRequestEvent;     //!< Resume 要求でシグナルするイベント
    nn::os::Event m_SuspendRequestEvent;    //!< Suspend 要求でシグナルするイベント
    nn::os::Mutex m_SuspendCounterLock;
    int64_t m_SuspendCounter; //!< Suspendで++, Resumeで--
    nn::os::Event m_SuspendCompletionEvent; //!< Suspend完了通知イベント

    const nn::TimeSpan m_IntervalTimeSpan;        //!< 最後の補正からどれだけ経過していれば時刻ダウンロードするか
    const nn::TimeSpan m_RetryTimeSpan;           //!< 自立補正処理の失敗時にリトライするまでの間隔
    const int m_ImmediateTryCountMax;             //!< ネットワーク接続を維持したまま自立補正処理を試行する最大回数
    const nn::TimeSpan m_ImmediateTryTimeSpan;    //!< ネットワーク接続を維持したまま自立補正処理を試行する間隔

    //!< スレッド管理のイベントへ終了をリクエスト
    void RequestStopThread() NN_NOEXCEPT
    {
        m_StopEvent.Signal();
    }

    //!< スレッド終了要求がきているかどうか
    bool IsStopThreadRequired() NN_NOEXCEPT
    {
        return m_StopEvent.TryWait();
    }

    static void ThreadFunction(void* p) NN_NOEXCEPT
    {
        reinterpret_cast<AutonomicEnsureNetworkClockAvailabilityThread*>(p)->ThreadFunctionImpl();
    }

    //!< ネットワーク時計が最後に補正された瞬間からの経過秒数取得
    nn::Result GetElapsedFromLastCorrection(int64_t* pOutSeconds) const NN_NOEXCEPT;
    //!< 次のタスク実行までの TimeSpan を取得
    nn::TimeSpan GetTimeSpanUntilTaskExecutionRequired() NN_NOEXCEPT;
    //!< 指定時間経過、もしくはトリガーされるまで待機(スレッド終了要求がきたら false を返す)
    bool WaitForNextExecution(const nn::TimeSpan& waitSpanToNext) NN_NOEXCEPT;
    //!< ネットワーク接続が確認できるまで待機(timeSpan 経過前にスレッド終了要求がきたら false を返す)
    bool WaitUntilNetworkConnectionConfirmed(nn::nifm::NetworkConnection* pNetworkConnection) NN_NOEXCEPT;
    //!< Resume まで待つ.(Suspendリクエストイベントシグナル時にしか呼べない)
    bool WaitForResume() NN_NOEXCEPT;
    //!< スレッド処理メイン
    void ThreadFunctionImpl() NN_NOEXCEPT;
    //!< 補正処理メイン
    nn::Result ExecuteNetworkClockCorrectionWithImmediateRetry(nn::nifm::NetworkConnection* pNetworkConnection) NN_NOEXCEPT;
    nn::Result ExecuteNetworkClockCorrectionImpl() NN_NOEXCEPT;
};

}}}} // nn::ntc::detail::service
