﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief nn::nsd の型定義(非公開)
 */

#pragma once

#include <nn/nsd/nsd_Common.h>
#include <nn/nsd/nsd_BackboneSettingsTypesPrivate.h>
#include <cstring>

#include <nn/sf/cmif/detail/sf_CmifAutogenInterfaceIncludes.h>
#include <nn/sf/detail/sf_AutogenInterfaceIncludes.h>
#include <nn/sf/impl/detail/sf_AutogenImplIncludes.h>

namespace nn { namespace nsd {

    const EnvironmentIdentifier EnvironmentIdentifierOfProductDevice = {"lp1"}; //!< 製品機で利用する環境識別子
    const EnvironmentIdentifier EnvironmentIdentifierOfNotProductDevice = {"dd1"}; //!< 製品機以外で利用する環境識別子

    /**
     * @brief   FQDN の変換ルールを表す構造体です。
     */
    struct FqdnEntry
    {
        Fqdn src; //!< 変換元
        Fqdn dst; //!< 変換先
    };

    /**
     *  @brief  基幹ネットワークサービスの設定群を表す構造体です。
     */
    struct BackboneSettings
    {
        static const int FqdnEntryCountMax = 16; //!< FQDN 変換ルールの最大数
        NasServiceSettings nasServiceSettings; //!< NAS 関連の設定群
        FqdnEntry fqdnEntries[FqdnEntryCountMax]; //!< FQDN 変換ルール群
    };

    /**
     * @brief   アプリケーション向けの(基幹サービスでない)ネットワークサービスの設定群を表す構造体です。
     */
    struct  ApplicationSettings
    {
        static const int FqdnEntryCountMax = 128; //!< FQDN 変換ルールの最大数
        FqdnEntry fqdnEntries[FqdnEntryCountMax]; //!< FQDN 変換ルール群
    };

    /**
     * @brief   デバイス ID を表す構造体
     * @details
     */
    struct DeviceId
    {
        static const size_t Size = 16; //!< デバイス ID を表す文字列のサイズ
        char value[Size]; //!< デバイス ID を表す文字列

        friend bool operator == (const DeviceId& lhs, const DeviceId& rhs) NN_NOEXCEPT
        {
            return std::memcmp(&lhs, &rhs, sizeof(lhs)) == 0;
        }

        friend bool operator != (const DeviceId& lhs, const DeviceId& rhs) NN_NOEXCEPT
        {
            return !(lhs == rhs);
        }
    };

    /**
     * @brief サービスディスカバリの設定をデバイスに保存する際の構造体
     */
    struct SaveData
    {
        uint32_t version; //!< バージョン
        DeviceId deviceId; //!< デバイスID
        SettingName settingName; //!< 設定名
        EnvironmentIdentifier environmentIdentifier; //!< 環境識別子
        int64_t expireUnixTime; //!< 設定の有効期限(UnixTime)
        BackboneSettings backboneSettings; //!< 基幹サービスの設定群
        ApplicationSettings applicationSettings; //!< アプリケーション向けサービスの設定群
        bool isAvailable; //!< 設定が有効か(時限失効するとfalse)
    };

    /**
     * @brief サービスディスカバリのサービスタイプを表します。
     */
    enum ServiceType
    {
        ServiceType_User  = 0,
        ServiceType_Admin = 1,
    };

    /**
     * @brief Result をやり取りするためのクラス
     */
    struct InnerResult
    {
        nn::Result result;
    };
}} // nn::nsd

namespace nn { namespace sf {
template <>
struct EnablesOut<nn::nsd::InnerResult> : public std::integral_constant<bool, true>
{
};
}}

