﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief メニュー向けの nn::nsd API
 */

#pragma once

#include <nn/nsd/nsd_Common.h>
#include <nn/nsd/nsd_DeleteMode.h>
#include <nn/nsd/nsd_ImportMode.h>

namespace nn { namespace nsd {

    const size_t WorkBufferSizeForImportSettings = 256 * 1024; //!< @ref ImportSettings() で必要になる作業用バッファのサイズです。

//! @name メニューが扱う API
//! @{

    /**
     * @brief サービスディスカバリの設定名を取得します。
     * @param[out]  pOut    設定名を格納するポインタ
     *
     * @retresult
     *      @handleresult{ResultNotFound, 設定が存在しません}
     * @endretresult
     *
     * @pre
     *  - pOut != nullptr
     * @post
     *  - pOut は NULL 終端している
     *
     * @details
     *  デバイスに登録されているサービスディスカバリの設定名を取得します。
     */
    nn::Result GetSettingName(SettingName* pOut) NN_NOEXCEPT;

    /**
     * @brief 環境識別子を取得します。
     * @param[out]  pOut    環境識別子を格納するポインタ
     *
     * @pre
     *  - pOut != nullptr
     * @post
     *  - pOut は NULL 終端している
     *
     * @details
     *  デバイスに登録されているサービスディスカバリの環境識別子を取得します。
     *  まだ登録されていない場合でも、開発機か量産機かを判定し、サービスディスカバリで
     *  利用されるデフォルト設定の環境識別子を返します。
     *
     *  ここで取得できる値を解釈したり、値に依存した実装を行うことは禁止とします。
     *
     *  具体的には、値を解釈せずにサーバーリクエストに埋め込んで送信したり、
     *  取得した環境識別子どうしの比較などは許容されますが、
     *  クライアントが持つ固有値と比較する("lp1" と等しいか、などの)実装は禁止します。
     *  返ってくる環境識別子の文字列は文字数も含め、将来変更となる可能性があります。
     */
     void GetEnvironmentIdentifier(EnvironmentIdentifier* pOut) NN_NOEXCEPT;

    /**
     * @brief サービスディスカバリの設定をデバイスから削除します。
     * @param[in]   deleteMode      削除するモード
     * @details
     *  指定の deleteMode に従ってデバイスに保存されてるサービスディスカバリの設定を削除します。
     */
    void DeleteSettings(DeleteMode deleteMode) NN_NOEXCEPT;

    /**
     * @class nn::nsd::ResultInvalidImportMode
     * @details
     *  この Result は @ref ImportSettings() で指定する @ref ImportMode で許可されなかった設定項目が変更されようとした場合に返ります。@n
     *  サーバーのレスポンスが意図しないものになっているか、 ImportSettings() への保存モードの指定ミスが考えられます。
     */

    /**
     * @brief サービスディスカバリの設定をデバイスへ保存します。
     * @param[in]   pBuffer             設定を指すバッファ
     * @param[in]   bufferSize          pBuffer のサイズ
     * @param[in]   pWorkBuffer         処理に必要な作業用バッファ
     * @param[in]   workBufferSize      pWorkBuffer のサイズ
     * @param[in]   importMode          インポートモード
     *
     * @retresult
     *      @handleresult{ResultInvalidSettingData, pBuffer が不正です}
     *      @handleresult{ResultInvalidImportMode, ImportMode が不正です}
     * @endretresult
     *
     * @pre
     *  - pBuffer != nullptr
     *  - workBufferSize >= @ref WorkBufferSizeForImportSettings
     *
     * @details
     *  入力の設定データ pBuffer を解析し、デバイスへ保存します。
     *
     *  指定の ImportMode で変更許可されていない項目が変更されようとした場合には、本 API は失敗します。
     */
    nn::Result ImportSettings(
        const void* pBuffer, size_t bufferSize,
        void* pWorkBuffer, size_t workBufferSize,
        ImportMode importMode) NN_NOEXCEPT;

    /**
     * @brief   環境識別子の変更を伴う設定インポートを不可能にします。
     * @details
     *  本関数の実行直後から、環境識別子を変更する設定インポート ( @ref ImportSettings() の実行 ) は必ず失敗するようになります。
     *
     *  設定は永続化され、本体のシャットダウンをまたいでも引き継がれます。
     *  本関数実行時、既に環境識別子の変更を伴う設定インポートが不可能な場合には何も行われません。

     */
    void DisableChangeEnvironmentIdentifier() NN_NOEXCEPT;

//! @}

}} // nn::nsd
