﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_StringUtil.h>

#include <nn/nsd/detail/util/nsd_PathHandler.h>
#include <nn/nsd/detail/util/nsd_StringUtility.h>
#include <nn/nsd/detail/json/nsd_JsonParser.h>
#include <nn/nsd/detail/jwt/nsd_JwtParser.h>
#include <nn/nsd/detail/nsd_Log.h>

namespace nn { namespace nsd { namespace detail { namespace json {

    // JWT の Header 向け rapidjson イベントハンドラ
    // Header はアプリによって左右されないので、 util 化する場合でもライブラリ層にあって良さそう
    class RapidJsonEventAccepterForHeader : public JsonEventAccepter
    {
        NN_DISALLOW_COPY(RapidJsonEventAccepterForHeader);
        NN_DISALLOW_MOVE(RapidJsonEventAccepterForHeader);

    private:
        jwt::JwtHeaderData* m_pOutJwtHeaderData;

        util::PathEntry m_EntryOfType; // typ key
        util::PathEntry m_EntryOfAlgorithm; // alg key

    public:
        explicit RapidJsonEventAccepterForHeader(jwt::JwtHeaderData* pOut) NN_NOEXCEPT:
            m_pOutJwtHeaderData(pOut),
            m_EntryOfType("/typ"),
            m_EntryOfAlgorithm("/alg")
        {
            NN_SDK_ASSERT_NOT_NULL(pOut);
            memset(pOut, 0, sizeof(jwt::JwtHeaderData));
        }
        virtual nn::Result AcceptEndObject(const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            if(pathHandler.GetPathDepth() == 0)
            {
                NN_RESULT_DO(m_EntryOfType.CheckAccepted());
                NN_RESULT_DO(m_EntryOfAlgorithm.CheckAccepted());
            }

            NN_RESULT_SUCCESS;
        }

        virtual nn::Result AcceptString(
            const char* value, size_t length, const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(length);
            // NN_SDK_LOG("  String:%s, len:%d copy:%d\n", str, length, copy);

            if(m_EntryOfType.Match(pathHandler))
            {
                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutJwtHeaderData->typ, value),
                    nn::nsd::ResultStringTooLong());
            }
            else if(m_EntryOfAlgorithm.Match(pathHandler))
            {
                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutJwtHeaderData->alg, value),
                    nn::nsd::ResultStringTooLong());
            }
            else
            {
                NN_DETAIL_NSD_ERROR("[NSD][DETAIL][JSON] RapidJsonEventAccepterForHeader / Unexpected json key string:%s\n", value);
                NN_RESULT_THROW(nn::nsd::ResultUnexpectedJsonItem());
            }

            NN_RESULT_SUCCESS;
        }

    };

}}}} // nn::nsd::detail::json

