﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nsd/nsd_Common.h>
#include <nn/nsd/detail/util/nsd_PathHandler.h>
#include <nn/nsd/detail/util/nsd_StringUtility.h>
#include <nn/nsd/detail/json/nsd_JsonParser.h>
#include <nn/nsd/nsd_TypesPrivate.h>
#include <nn/nsd/detail/nsd_Log.h>

#include <nn/util/util_IntUtil.h>

namespace nn { namespace nsd { namespace detail { namespace json {

    // JWT の Calims Set の JSON ドキュメント解析 rapidjson イベントハンドラ。
    // アプリによって解析内容が様々なので JWT 解析処理を util 化するならば、
    // これはアプリレイヤーで定義されるべきものです。
    class RapidJsonEventAccepterForClaims : public nn::nsd::detail::json::JsonEventAccepter
    {
        NN_DISALLOW_COPY(RapidJsonEventAccepterForClaims);
        NN_DISALLOW_MOVE(RapidJsonEventAccepterForClaims);

    private:
        // BackboneSettings/Nas/ の設定を入れるクラス
        class NasServiceSettingsBox
        {
        private:
            // BackboneSettings/Nas/AnyService 直下の NAS サービスエントリのアイテムの存在確認フラグ
            struct NasServiceEntryFlag
            {
                bool hasServiceName;
                bool hasClientId;
                bool hasRedirectUri;

                NasServiceEntryFlag()
                {
                    this->Reset();
                }

                void Reset()
                {
                    hasServiceName = false;
                    hasClientId = false;
                    hasRedirectUri = false;
                }
            };
            NasServiceEntryFlag m_NasServiceEntryFlag;

            NasServiceSettings* m_pOutNasServiceSettings;
            int m_CurrentIndex;

        public:
            explicit NasServiceSettingsBox(NasServiceSettings* pOut) NN_NOEXCEPT:
                m_pOutNasServiceSettings(pOut),
                m_CurrentIndex(0)
            {
            }

            nn::Result PushNasRequestFqdn(const char* str) NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutNasServiceSettings->nasRequestServiceFqdn, str),
                    nn::nsd::ResultStringTooLong());

                NN_RESULT_SUCCESS;
            }

            nn::Result PushNasApiFqdn(const char* str) NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutNasServiceSettings->nasApiServiceFqdn, str),
                    nn::nsd::ResultStringTooLong());

                NN_RESULT_SUCCESS;
            }

            nn::Result PushNasServiceName(const char* str) NN_NOEXCEPT
            {
                NN_RESULT_DO( CheckNasServiceCapacity() );

                // 同じ名前のエントリーがすでにあったらエラー
                for(int i = 0 ; i < m_CurrentIndex ; ++i)
                {
                    if(util::IsEqualString(m_pOutNasServiceSettings->entries[i].nasServiceName, str))
                    {
                        NN_RESULT_THROW(nn::nsd::ResultMultipleItem());
                    }
                }

                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutNasServiceSettings->entries[m_CurrentIndex].nasServiceName, str),
                    nn::nsd::ResultStringTooLong());

                m_NasServiceEntryFlag.hasServiceName = true;
                NN_RESULT_SUCCESS;
            }

            nn::Result PushNasServiceClientId(uint64_t value) NN_NOEXCEPT
            {
                NN_RESULT_DO( CheckNasServiceCapacity() );

                m_pOutNasServiceSettings->entries[m_CurrentIndex].nasServiceSetting.clientId = value;

                m_NasServiceEntryFlag.hasClientId = true;
                NN_RESULT_SUCCESS;
            }

            nn::Result PushNasServiceRedirectUri(const char* str) NN_NOEXCEPT
            {
                NN_RESULT_DO( CheckNasServiceCapacity() );

                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutNasServiceSettings->entries[m_CurrentIndex].nasServiceSetting.redirectUri, str),
                    nn::nsd::ResultStringTooLong());

                m_NasServiceEntryFlag.hasRedirectUri = true;
                NN_RESULT_SUCCESS;
            }

            Result EndNasServiceEntry() NN_NOEXCEPT
            {
                NN_RESULT_DO( CheckNasServiceCapacity() );

                // 以下それぞれ必須項目
                NN_RESULT_THROW_UNLESS(
                    m_NasServiceEntryFlag.hasServiceName &&
                    m_NasServiceEntryFlag.hasClientId &&
                    m_NasServiceEntryFlag.hasRedirectUri,
                    nn::nsd::ResultRequiredItemNotFound());

                m_CurrentIndex++;
                m_NasServiceEntryFlag.Reset();

                NN_RESULT_SUCCESS;
            }

            Result Finish() NN_NOEXCEPT
            {
                // 現時点では NxAccount が必須項目

                //
                // TODO:NxShopも必須にする
                //

                bool found = false;
                for(auto& entry : m_pOutNasServiceSettings->entries)
                {
                    if(entry.nasServiceName == NasServiceNameOfNxAccount)
                    {
                        found = true;
                        break;
                    }
                }
                NN_RESULT_THROW_UNLESS(found, nn::nsd::ResultRequiredItemNotFound());

                NN_RESULT_SUCCESS;
            }

        private:
            nn::Result CheckNasServiceCapacity() const NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(
                    m_CurrentIndex < static_cast<int>(m_pOutNasServiceSettings->NasServiceEntryCountMax),
                    nn::nsd::ResultBackbonesettingsNasServiceOverCapacity());
                NN_RESULT_SUCCESS;
            }
        };

        // FqdnEntries を入れるクラス
        // BackboneSettings/FqdnEntries/, ApplicationSettings/FqdnEntries/ の両方で使う
        class FqdnEntriesBox
        {
        private:
            nn::nsd::FqdnEntry* m_pOutFqdnEntries;
            const int m_OutFqdnEntriesCount;
            int m_CurrentIndex;
            int m_PushCount;
        public:
            FqdnEntriesBox(nn::nsd::FqdnEntry* pOutFqdnEntries, int outFqdnEntriesCount) NN_NOEXCEPT:
                m_pOutFqdnEntries(pOutFqdnEntries),
                m_OutFqdnEntriesCount(outFqdnEntriesCount),
                m_CurrentIndex(0),
                m_PushCount(0)
            {
                NN_SDK_ASSERT_NOT_NULL(pOutFqdnEntries);
            }

            Result Next() NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(m_PushCount == 2, nn::nsd::ResultInvalidFqdnEntryFormat());
                m_PushCount = 0;
                m_CurrentIndex++;
                NN_RESULT_SUCCESS;
            }

            Result Finish() NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(m_PushCount == 0, nn::nsd::ResultInvalidFqdnEntryFormat());
                NN_RESULT_SUCCESS;
            }

            Result Push(const char* str) NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(
                    m_CurrentIndex < m_OutFqdnEntriesCount,
                    nn::nsd::ResultFqdnEntryOverCapacity() );

                if(m_PushCount == 0) // src
                {
                    NN_RESULT_THROW_UNLESS(
                        util::SetString(&m_pOutFqdnEntries[m_CurrentIndex].src, str),
                        nn::nsd::ResultStringTooLong());
                }
                else if(m_PushCount == 1) // dst
                {
                    NN_RESULT_THROW_UNLESS(
                        util::SetString(&m_pOutFqdnEntries[m_CurrentIndex].dst, str),
                        nn::nsd::ResultStringTooLong());
                }
                else // error
                {
                    NN_RESULT_THROW( nn::nsd::ResultInvalidFqdnEntryFormat() );
                }
                m_PushCount++;
                NN_RESULT_SUCCESS;
            }
        };

        // ApplicationSettings/ 直下のアイテムを入れるクラス
        class ApplicationSettingsBox
        {
        private:
            nn::nsd::ApplicationSettings* m_pApplicationSettings;

        public:
            FqdnEntriesBox m_FqdnEntriesBox;

            explicit ApplicationSettingsBox(nn::nsd::ApplicationSettings* pOut) NN_NOEXCEPT:
                m_pApplicationSettings(pOut),
                m_FqdnEntriesBox(pOut->fqdnEntries, pOut->FqdnEntryCountMax)
            {
                NN_UNUSED(m_pApplicationSettings);
            }


        };

        // BackboneSettings/ 直下のアイテムを入れるクラス
        class BackboneSettingsBox
        {
        public:
            NasServiceSettingsBox m_NasServiceSettingsBox;
            FqdnEntriesBox m_FqdnEntriesBox;

            explicit BackboneSettingsBox(nn::nsd::BackboneSettings* pOut) NN_NOEXCEPT:
                m_NasServiceSettingsBox(&pOut->nasServiceSettings),
                m_FqdnEntriesBox(pOut->fqdnEntries, pOut->FqdnEntryCountMax)
            {
                NN_SDK_ASSERT_NOT_NULL(pOut);
            }
        };

        // サーバーエラーを入れるクラス
        class ServerErrorResponseBox
        {
        public:
            bool m_HasError;

            struct ErrorCodeStr
            {
                static const size_t Size = 16;
                char value[Size]; // サイズ確認
            } m_ErrorCode;

            struct ErrorMessage
            {
                static const size_t Size = 64;
                char value[Size]; // サイズ確認
            } m_ErrorMessage;

            ServerErrorResponseBox() NN_NOEXCEPT:
                m_HasError(false)
            {
                util::ClearString(&m_ErrorCode);
                util::ClearString(&m_ErrorMessage);
            }

            bool HasError() const NN_NOEXCEPT
            {
                return m_HasError;
            }

            const char* GetErrorCode() const NN_NOEXCEPT
            {
                return m_ErrorCode.value;
            }

            const char* GetErrorMessage() const NN_NOEXCEPT
            {
                return m_ErrorMessage.value;
            }

            bool SetErrorCode(const char* str, size_t length) NN_NOEXCEPT
            {
                NN_UNUSED(length);
                m_HasError = true;
                return util::SetString(&m_ErrorCode, str);
            }

            bool SetErrorMessage(const char* str, size_t length) NN_NOEXCEPT
            {
                NN_UNUSED(length);
                return util::SetString(&m_ErrorMessage, str);
            }
        };

        nn::nsd::SaveData* m_pOutSaveData;
        ApplicationSettingsBox m_ApplicationSettingsBox;
        BackboneSettingsBox m_BackboneSettingsBox;
        ServerErrorResponseBox m_ServerErrorResponseBox;

        struct RootEntry
        {
            RootEntry():
                version("/Version"),
                settingName("/SettingName"),
                environmentIdentifier("/Environment"),
                deviceId("/DeviceId"),
                expire("/Expire")
            {}

            util::PathEntry version;
            util::PathEntry settingName;
            util::PathEntry environmentIdentifier;
            util::PathEntry deviceId;
            util::PathEntry expire;

            struct ErrorEntry
            {
                ErrorEntry():
                    errorCode("/errors[/code"),
                    errorMessage("/errors[/message")
                {}
                util::PathEntry errorCode;
                util::PathEntry errorMessage;
            };
            ErrorEntry errorEntry;
        };
        RootEntry m_RootEntry;

        struct BackboneSettingsEntry
        {
            BackboneSettingsEntry():
                fqdnEntries("/BackboneSettings/FqdnEntries[[")
            {}

            struct Nas
            {
                Nas():
                    requestFqdn("/BackboneSettings/Nas/RequestFqdn"),
                    apiFqdn("/BackboneSettings/Nas/ApiFqdn")
                {}

                util::PathEntry requestFqdn;
                util::PathEntry apiFqdn;

                /*
                struct NxAccount
                {
                    NxAccount():
                        clientId("/BackboneSettings/Nas/NxAccount/ClientId"),
                        redirectUri("/BackboneSettings/Nas/NxAccount/RedirectUri")
                    {}
                    util::PathEntry clientId;
                    util::PathEntry redirectUri;
                };
                NxAccount nxAccount;

                struct NxShop
                {
                    NxShop():
                        clientId("/BackboneSettings/Nas/NxShop/ClientId"),
                        redirectUri("/BackboneSettings/Nas/NxShop/RedirectUri")
                    {}
                    util::PathEntry clientId;
                    util::PathEntry redirectUri;
                };
                NxShop nxShop;
                */
            };
            Nas nasEntry;
            util::PathEntry fqdnEntries;
        };
        BackboneSettingsEntry m_BackboneSettingsEntry;

        struct ApplicationSettingsEntry
        {
            ApplicationSettingsEntry():
                fqdnEntries("/ApplicationSettings/FqdnEntries[[")
            {}
            util::PathEntry fqdnEntries;
        };
        ApplicationSettingsEntry m_ApplicationSettingsEntry;

    public:
        explicit RapidJsonEventAccepterForClaims(nn::nsd::SaveData *pOut) NN_NOEXCEPT:
                m_pOutSaveData(pOut),
                m_ApplicationSettingsBox(&pOut->applicationSettings),
                m_BackboneSettingsBox(&pOut->backboneSettings)
        {
            NN_SDK_ASSERT_NOT_NULL(pOut);
            std::memset(pOut, 0, sizeof(nn::nsd::SaveData));
        }

        bool HasServerError() const NN_NOEXCEPT
        {
            return m_ServerErrorResponseBox.HasError();
        }

        const char* GetServerErrorCode() const NN_NOEXCEPT
        {
            return m_ServerErrorResponseBox.GetErrorCode();
        }

        const char* GetServerErrorMessage() const NN_NOEXCEPT
        {
            return m_ServerErrorResponseBox.GetErrorMessage();
        }

        virtual nn::Result AcceptUint64(
            uint64_t value, const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            if(m_RootEntry.version.Match(pathHandler))
            {
                NN_RESULT_THROW_UNLESS(
                    nn::util::IsIntValueRepresentable<uint32_t>(value),
                    nn::nsd::ResultOutOfRangeValue());
                m_pOutSaveData->version = static_cast<uint32_t>(value);
            }
            else if(m_RootEntry.expire.Match(pathHandler))
            {
                NN_RESULT_DO(SetExpire(value));
            }
            else if(pathHandler.CompareFront("/BackboneSettings/Nas/", 4)) // /BackboneSettings/Nas/*/*
            {
                if(pathHandler.CompareTail("/ClientId"))
                {
                    NN_RESULT_DO(m_BackboneSettingsBox.m_NasServiceSettingsBox.PushNasServiceClientId(value));
                }
            }
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result AcceptInt64(
            int64_t value, const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            // マイナスのとき呼ばれる
            if(m_RootEntry.expire.Match(pathHandler))
            {
                NN_RESULT_DO(SetExpire(value));
            }
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result AcceptString(
            const char* value, size_t length, const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            if(m_RootEntry.settingName.Match(pathHandler))
            {
                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutSaveData->settingName, value),
                    nn::nsd::ResultStringTooLong());
            }
            else if(m_RootEntry.environmentIdentifier.Match(pathHandler))
            {
                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutSaveData->environmentIdentifier, value),
                    nn::nsd::ResultStringTooLong());
            }
            else if(m_RootEntry.deviceId.Match(pathHandler))
            {
                NN_RESULT_THROW_UNLESS(
                    util::SetString(&m_pOutSaveData->deviceId, value),
                    nn::nsd::ResultStringTooLong());
            }
            else if(m_BackboneSettingsEntry.nasEntry.requestFqdn.Match(pathHandler))
            {
                NN_RESULT_DO(m_BackboneSettingsBox.m_NasServiceSettingsBox.PushNasRequestFqdn(value));
            }
            else if(m_BackboneSettingsEntry.nasEntry.apiFqdn.Match(pathHandler))
            {
                NN_RESULT_DO(m_BackboneSettingsBox.m_NasServiceSettingsBox.PushNasApiFqdn(value));
            }
            else if(pathHandler.CompareFront("/BackboneSettings/Nas/", 4)) // /BackboneSettings/Nas/*/*
            {
                if(pathHandler.CompareTail("/RedirectUri"))
                {
                    NN_RESULT_DO(m_BackboneSettingsBox.m_NasServiceSettingsBox.PushNasServiceRedirectUri(value));
                }
            }
            else if(m_BackboneSettingsEntry.fqdnEntries.Match(pathHandler))
            {
                NN_RESULT_DO(m_BackboneSettingsBox.m_FqdnEntriesBox.Push(value));
            }
            else if(m_ApplicationSettingsEntry.fqdnEntries.Match(pathHandler))
            {
                NN_RESULT_DO(m_ApplicationSettingsBox.m_FqdnEntriesBox.Push(value));
            }
            else if(m_RootEntry.errorEntry.errorCode.Match(pathHandler))
            {
                m_ServerErrorResponseBox.SetErrorCode(value, length);
            }
            else if(m_RootEntry.errorEntry.errorMessage.Match(pathHandler))
            {
                m_ServerErrorResponseBox.SetErrorMessage(value, length);
            }
            else
            {
                NN_DETAIL_NSD_ERROR("[NSD][DETAIL][JSON] RapidJsonEventAccepterForClaims / Unexpected json key string:%s\n", value);
                NN_RESULT_THROW(nn::nsd::ResultUnexpectedJsonItem());
            }
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result AcceptJsonKey(
            const char* key, size_t length, const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(length);

            if(pathHandler.CompareFront("/BackboneSettings/Nas/", 3)) // /BackboneSettings/Nas/*
            {
                // /BackboneSettings/Nas/RequestFqdn, /BackboneSettings/Nas/ApiFqdn 以外は NAS サービス名になる.
                // NAS サービス名は任意の文字列を許容する.
                if( !m_BackboneSettingsEntry.nasEntry.requestFqdn.Match(pathHandler) &&
                    !m_BackboneSettingsEntry.nasEntry.apiFqdn.Match(pathHandler))
                {
                    NN_RESULT_DO(m_BackboneSettingsBox.m_NasServiceSettingsBox.PushNasServiceName(key));
                }
            }

            NN_RESULT_SUCCESS;
        }

        virtual nn::Result AcceptEndObject(const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            if(pathHandler.CompareFront("/BackboneSettings/Nas/", 3)) // /BackboneSettings/Nas/* の終了
            {
                NN_RESULT_DO(m_BackboneSettingsBox.m_NasServiceSettingsBox.EndNasServiceEntry());
            }
            else if(pathHandler.Compare("/BackboneSettings/Nas")) // BackboneSettings/Nas 設定の終了
            {
                NN_RESULT_DO(m_BackboneSettingsBox.m_NasServiceSettingsBox.Finish());
            }

            if(pathHandler.GetPathDepth() == 0) // JSON ドキュメントの終了
            {
                // サーバーエラーが発生していなければ、必須項目を確認する
                if(!m_ServerErrorResponseBox.HasError())
                {
                    NN_RESULT_DO(CheckRequriedItem());
                }
            }

            NN_RESULT_SUCCESS;
        }

        virtual nn::Result AcceptEndArray(const util::PathHandler& pathHandler) NN_NOEXCEPT NN_OVERRIDE
        {
            if(pathHandler.GetArrayDepth() == 1 || pathHandler.GetArrayDepth() == 2)
            {
                if(m_BackboneSettingsEntry.fqdnEntries.Match(pathHandler))
                {
                    NN_RESULT_DO(m_BackboneSettingsBox.m_FqdnEntriesBox.Next()); // 次の FQDN 読み込みへ
                }
                else if(pathHandler.Compare("BackboneSettings/FqdnEntries["))
                {
                    NN_RESULT_DO(m_BackboneSettingsBox.m_FqdnEntriesBox.Finish()); // FQDN エントリー終了
                }
                else if(m_ApplicationSettingsEntry.fqdnEntries.Match(pathHandler))
                {
                    NN_RESULT_DO(m_ApplicationSettingsBox.m_FqdnEntriesBox.Next()); // 次の FQDN 読み込みへ
                }
                else if(pathHandler.Compare("BackboneSettings/FqdnEntries["))
                {
                    NN_RESULT_DO(m_ApplicationSettingsBox.m_FqdnEntriesBox.Finish()); // FQDN エントリー終了
                }
            }

            NN_RESULT_SUCCESS;
        }

    private:
        nn::Result CheckRequriedItem() const NN_NOEXCEPT
        {
            NN_RESULT_DO(m_RootEntry.version.CheckAccepted());
            NN_RESULT_DO(m_RootEntry.settingName.CheckAccepted());
            NN_RESULT_DO(m_RootEntry.environmentIdentifier.CheckAccepted());
            NN_RESULT_DO(m_RootEntry.deviceId.CheckAccepted());
            NN_RESULT_DO(m_RootEntry.expire.CheckAccepted());

            NN_RESULT_DO(m_BackboneSettingsEntry.fqdnEntries.CheckAccepted());
            NN_RESULT_DO(m_BackboneSettingsEntry.nasEntry.apiFqdn.CheckAccepted());
            NN_RESULT_DO(m_BackboneSettingsEntry.nasEntry.requestFqdn.CheckAccepted());

            // NN_RESULT_DO(m_ApplicationSettingsEntry.fqdnEntries.CheckAccepted()); // アプリ向けはなくてもいい

            NN_RESULT_SUCCESS;
        }

        template <typename Int>
        Result SetExpire(Int value) NN_NOEXCEPT
        {
            NN_STATIC_ASSERT( std::is_integral<Int>::value);

            NN_RESULT_THROW_UNLESS(
                nn::util::IsIntValueRepresentable<int64_t>(value),
                nn::nsd::ResultOutOfRangeValue());

            m_pOutSaveData->expireUnixTime = static_cast<int64_t>(value);

            NN_RESULT_SUCCESS;
        }
    };

}}}} // nn::nsd::detail::json

