﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Mutex.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>

#include <atomic>

namespace nn { namespace ns { namespace srv {
    class VulnerabilityManager
    {
    public:
        enum class NeedsUpdatePolicy
        {
            Default = 0,
            ForceTrue,
            ForceFalse,

            PolicyCount,
        };

        enum class StorePolicy
        {
            OnStorage = 0,
            OnMemory = 1,
        };

        VulnerabilityManager() NN_NOEXCEPT : m_NeedsUpdateVulnerability(false), m_NeedsUpdatePolicy(NeedsUpdatePolicy::Default), m_CachedSystemUpdateMetaKey({}), m_SystemUpdateMetaKeyLock(false), m_EnableDebug(false)
        {}
        void Initialize() NN_NOEXCEPT
        {
            InitializeImpl(StorePolicy::OnStorage);
        }
        void InitializeForTest(StorePolicy storePolicy) NN_NOEXCEPT
        {
            InitializeImpl(storePolicy);
        }
        void Finalize() NN_NOEXCEPT;

        Result UpdateSafeSystemVersionInfo(nn::ncm::SystemUpdateId systemUpdateId, uint32_t version) NN_NOEXCEPT;
        Result UpdateSafeSystemVersionForDebug(nn::ncm::SystemUpdateId systemUpdateId, uint32_t version) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(m_EnableDebug, ResultNotPermittedOnProduction());
            return UpdateSafeSystemVersionInfo(systemUpdateId, version);
        }
        bool NeedsUpdateVulnerability() NN_NOEXCEPT;
        void GetSafeSystemVersion(ncm::ContentMetaKey* out) NN_NOEXCEPT
        {
            *out = m_CachedSystemUpdateMetaKey;
        }
    private:
        void InitializeImpl(StorePolicy storePolicy) NN_NOEXCEPT;
        bool NeedsUpdate(const ncm::ContentMetaKey& latest) NN_NOEXCEPT;
        void CacheSystemUpdateMeta(const ncm::ContentMetaKey& meta) NN_NOEXCEPT;
        Result WriteSystemUpdateMetaKey(const ncm::ContentMetaKey& entry) NN_NOEXCEPT;
        Result ReadSystemUpdateMetaKey(ncm::ContentMetaKey* out) NN_NOEXCEPT;
        Result InitializeNeedsUpdateVulnerabilityFlag() NN_NOEXCEPT;
        Result UpdateNeedsUpdateVulnerabilityFlag(const ncm::ContentMetaKey& entry) NN_NOEXCEPT;
        Result MountSaveData() NN_NOEXCEPT;

        std::atomic<bool> m_NeedsUpdateVulnerability;
        NeedsUpdatePolicy m_NeedsUpdatePolicy;
        ncm::ContentMetaKey m_CachedSystemUpdateMetaKey;
        os::Mutex m_SystemUpdateMetaKeyLock;
        bool m_EnableDebug;
        StorePolicy m_StorePolicy;
    };
}}}
