﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/os/os_Thread.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <mutex>

namespace nn { namespace ns { namespace srv {

    class RequestList
    {
    public:
        enum class ProcessType
        {
            All, // 全処理
            Offline, // オフラインで実行可能な処理
            Lightweight, // アプリの裏でも動かせる軽量な処理(オンラインを必要とするものを含む)
        };

        // type の条件に合致する処理を持っているかどうかを返す
        virtual bool NeedsProcess(ProcessType type) NN_NOEXCEPT = 0;

        // type の条件に合致する処理を開始する
        virtual Result Prepare(ProcessType type) NN_NOEXCEPT = 0;

        // Prepare によって開始された処理を持っているかどうかを返す
        virtual bool HasPreparedRequest() NN_NOEXCEPT = 0;

        // ManualClear イベントのみ利用できる
        // AutoClear であっても Wait はできるが、その後イベントが発生したかを判定できなくなってしまう
        virtual void WaitAlongWith(os::SystemEvent** events, int numEvents) NN_NOEXCEPT = 0;

        // エラーが発生しなかったら true、何かエラーがあったら false が返る
        // 内部実装として、ネットワーク接続が必要なタスク処理中であれば、これを呼ぶときはネットワークは接続中とする
        // (ネットワーク処理中なのに接続されていない場合は、Cleanup の方を呼ぶ)
        virtual bool HandleDone() NN_NOEXCEPT = 0;
        virtual void Cleanup() NN_NOEXCEPT = 0;
    };

    class RequestServer
    {
    public:
        RequestServer(RequestList* list, uint8_t* stack, size_t stackSize) NN_NOEXCEPT : m_IsStarted(), m_StopCount(), m_List(list), m_Stack(stack), m_StackSize(stackSize){}
        ~RequestServer() NN_NOEXCEPT;
        void Initialize(TimeSpan retryAfter) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        bool HasRequest() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_HasRequestLock);

            return m_HasRequest;
        }

        os::SystemEvent& GetHasRequestChangedEvent() NN_NOEXCEPT
        {
            return m_HasRequestChangedEvent;
        }

        class ManagedStop
        {
            NN_DISALLOW_COPY(ManagedStop);

        public:
            ManagedStop() NN_NOEXCEPT : m_Server() {}
            explicit ManagedStop(RequestServer* server) NN_NOEXCEPT : m_Server(server) {}

            ManagedStop(ManagedStop&& rvalue) NN_NOEXCEPT
            {
                m_Server = rvalue.m_Server;
                rvalue.m_Server = nullptr;
            }

            ManagedStop& operator=(ManagedStop&& rvalue) NN_NOEXCEPT
            {
                ManagedStop(std::move(rvalue)).swap(*this);
                return *this;
            }

            void swap(ManagedStop& other) NN_NOEXCEPT
            {
                std::swap(m_Server, other.m_Server);
            }

            ~ManagedStop() NN_NOEXCEPT
            {
                if (m_Server)
                {
                    m_Server->Unstop();
                }
            }

            ManagedStop AddReference() NN_NOEXCEPT
            {
                return m_Server->Stop();
            }

        private:
            RequestServer* m_Server;
        };

        ManagedStop Stop() NN_NOEXCEPT;


    private:
        void Run() NN_NOEXCEPT;
        void Unstop() NN_NOEXCEPT;

        void StartImpl() NN_NOEXCEPT;
        void StopImpl() NN_NOEXCEPT;

        void UpdateHasRequest(bool value) NN_NOEXCEPT;

        bool m_IsStarted;
        int m_StopCount;
        bool m_HasRequest;
        ManualClearSystemEvent m_StopEvent;
        ManualClearSystemEvent m_HasRequestChangedEvent;
        NonRecursiveMutex m_Mutex;
        NonRecursiveMutex m_HasRequestLock;

        RequestList* m_List;
        os::ThreadType m_Thread;
        uint8_t* m_Stack;
        size_t m_StackSize;

        TimeSpan m_RetryAfter;
    };
}}}
