﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ns/ns_NotificationSystemApi.h>
#include <nn/ns/srv/ns_ApplicationInstallRequest.h>
#include <nn/ns/srv/ns_AsyncTaskManager.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/ns/srv/ns_VulnerabilityManager.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/nim/nim_ETag.h>
#include <functional>

#define NN_PN_TRACE(...) NN_DETAIL_NS_TRACE("[PushNotificationDispatcher] " __VA_ARGS__)

namespace nn { namespace ns { namespace srv {

    class PushNotificationDispatcher
    {
        NN_DISALLOW_COPY(PushNotificationDispatcher);
        NN_DISALLOW_MOVE(PushNotificationDispatcher);

    public:

        struct Config
        {
            bool        isNotificationRegistrationEnabled;
            bool        isRequestOnColdBootEnabled;
            uint32_t    retryIntervalMin;
            uint32_t    retryIntervalMax;
        };

        PushNotificationDispatcher(uint8_t* stack, size_t stackSize, uint8_t* notificationTaskManagerStack, size_t notificationTaskManagerStackSize) NN_NOEXCEPT
            :   m_Stack(stack), m_StackSize(stackSize),
                m_DispatchThreadRunning(false),
                m_DispatchThreadLock(false), m_ListenTarget(0),
                m_Config({}), m_AsyncTaskManager(notificationTaskManagerStack, notificationTaskManagerStackSize), m_VulnerabilityManager(nullptr)
        {}

        Result Initialize(nn::Bit64 listenTarget, ApplicationInstallRequestList* requestList, RequestServer* requestServer, VulnerabilityManager* vulnerabilityManager, ApplicationVersionManager* applicationVersionManager) NN_NOEXCEPT;
        Result Initialize(nn::Bit64 listenTarget, Config& config, ApplicationInstallRequestList* requestList, RequestServer* requestServer, VulnerabilityManager* vulnerabilityManager, ApplicationVersionManager* applicationVersionManager, AsyncTaskManager::NotificationHandlerHolder handlers[], int handlersCount, AsyncTaskManager::Config& ntmConfig) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        Result RequestDownloadTaskList(const nn::nim::ETag& eTag, bool notifiesRequiredSystemUpdate) NN_NOEXCEPT;
        Result RequestVersionList(const nn::nim::ETag& eTag) NN_NOEXCEPT;

        Result IsNotificationSetupCompleted(sf::Out<bool> outValue) NN_NOEXCEPT;
        Result GetLastNotificationInfoCount(sf::Out<int64_t> outValue) NN_NOEXCEPT;
        Result ListLastNotificationInfo(sf::Out<int> outCount, const sf::OutArray<ns::NotificationInfo>& outList) NN_NOEXCEPT;
        Result ListNotificationTask(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ns::AsyncTask>& outList) NN_NOEXCEPT
        {
            return m_AsyncTaskManager.ListNotificationTask(outCount, outList);
        }

        // ShutdownManager からアクセスするため
        void Stop() NN_NOEXCEPT;

    private:
        enum EventIndex {
            StopEvent,
            NotificationEvent,
            NumberOfEvent,
        };

        Result InitializeCommon(nn::Bit64 listenTarget, const Config& config, VulnerabilityManager* vulnerabilityManager) NN_NOEXCEPT;
        void Start() NN_NOEXCEPT;
        void Dispatch() NN_NOEXCEPT;
        void DispatchThreadFunc() NN_NOEXCEPT;
        void SetupNotifications() NN_NOEXCEPT;
        void ProcessBeforeWaiting() NN_NOEXCEPT;

        static Config MakeDefaultConfig() NN_NOEXCEPT;
        EventIndex WaitAnyEvents() NN_NOEXCEPT;
        void PushLastInfo(const NotificationInfo& info) NN_NOEXCEPT;

        os::ThreadType m_Thread;
        os::SystemEvent m_NotificationEvent;
        ManualClearSystemEvent m_StopEvent;
        MultiWaitSystemEvent m_MultiWait;

        uint8_t* m_Stack;
        size_t m_StackSize;

        bool m_DispatchThreadRunning;
        std::atomic_bool m_SetupCompleted{};
        nn::os::Mutex m_DispatchThreadLock;
        nn::Bit64 m_ListenTarget;

        Config m_Config;

        NonRecursiveMutex m_LastInfoMutex;
        static const int MaxLastInfoCount = 256;
        int64_t m_LastInfoCount;
        NotificationInfo m_LastInfoList[MaxLastInfoCount];

        AsyncTaskManager m_AsyncTaskManager;
        VulnerabilityManager* m_VulnerabilityManager;
    };
}}}
