﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <utility>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/ns/srv/ns_OsUtil.h>

namespace nn { namespace ns { namespace srv {

    class LockedBufferManager
    {
    public:
        LockedBufferManager(void* buffer, size_t size) NN_NOEXCEPT : m_pBuffer(buffer), m_Size(size) {}

        class LockedBuffer
        {
            NN_DISALLOW_COPY(LockedBuffer);

        public:
            LockedBuffer(LockedBufferManager* pManager, void* buffer, size_t size) NN_NOEXCEPT
                : m_pManager(pManager), m_pBuffer(buffer), m_Size(size)
            {
            }

            LockedBuffer(LockedBuffer&& rvalue) NN_NOEXCEPT
            {
                *this = std::move(rvalue);
            }

            LockedBuffer& operator=(LockedBuffer&& rvalue) NN_NOEXCEPT
            {
                LockedBuffer(std::move(rvalue)).swap(*this);
                return *this;
            }

            ~LockedBuffer() NN_NOEXCEPT
            {
                if (m_pManager)
                {
                    m_pManager->Release();
                }
            }

            template <class T>
            T Get() NN_NOEXCEPT { return reinterpret_cast<T>(m_pBuffer); }
            size_t GetSize() NN_NOEXCEPT { return m_Size; }

            void swap(LockedBuffer& other) NN_NOEXCEPT
            {
                std::swap(m_pBuffer, other.m_pBuffer);
                std::swap(m_Size, other.m_Size);
                std::swap(m_pManager, m_pManager);
            }

        private:
            LockedBufferManager* m_pManager;
            void* m_pBuffer;
            size_t m_Size;
        };

        LockedBuffer Allocate(size_t requiredSize) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(requiredSize <= m_Size);
            m_Mutex.lock();
            return LockedBuffer(this, m_pBuffer, m_Size);
        }

    private:
        void Release() NN_NOEXCEPT
        {
            m_Mutex.unlock();
        }

        NonRecursiveMutex m_Mutex;
        void* m_pBuffer;
        size_t m_Size;
    };

}}}
