﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/ns/srv/ns_GameCardInfo.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_AppletLauncher.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_ApplicationControlDataManager.h>
#include <nn/ns/detail/ns_IApplicationManagerInterface.sfdl.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Event.h>
#include <nn/util/util_Optional.h>
#include "ns_OsUtil.h"

namespace nn { namespace ns {
    struct GameCardAttachmentInfo;
}}

namespace nn { namespace ns { namespace srv {
    class GameCardStopperImpl;

    class GameCardManager
    {
        friend class GameCardStopperImpl;
    public:
        GameCardManager(uint8_t* stack, size_t stackSize) NN_NOEXCEPT :
            m_IsCardAttached(), m_IsCardUpdateAppliedOnAutoBoot(), m_IsSignalEventDisabledOnRemove(), m_Thread(), m_Stack(stack), m_StackSize(stackSize), m_StateLock(true), m_ThreadLock(false), m_EventCount(), m_StopEvent(os::EventClearMode_AutoClear), m_Stopper() {}

        Result Initialize(IntegratedContentManager* contentManager, ApplicationRecordDatabase* recordDb, ApplicationControlDataManager* dataManager, AppletLauncher* appletLauncher) NN_NOEXCEPT;

        util::optional<ncm::ApplicationId> GetApplicationIdForAutoBoot() NN_NOEXCEPT
        {
            return m_ApplicationIdForAutoBoot;
        }

        os::NativeHandle GetAttachmentEvent() NN_NOEXCEPT;

        os::NativeHandle GetMountFailureEvent() NN_NOEXCEPT;

        void GetAttachmentInfo(GameCardAttachmentInfo* pOutInfo) NN_NOEXCEPT;

        os::SystemEvent& GetUpdateDetectionEvent() NN_NOEXCEPT
        {
            return m_UpdateDetectionEvent;
        }

        void Run() NN_NOEXCEPT;

        void ApplyCardUpdateForAutoBoot() NN_NOEXCEPT;

        sf::SharedPointer<ns::detail::IGameCardStopper> GetStopper() NN_NOEXCEPT;

        Result ImportTicket(const fs::RightsId& id) NN_NOEXCEPT;

        Result GetGameCardInfo(GameCardInfo* out) NN_NOEXCEPT;

        bool IsGameCardInserted() const NN_NOEXCEPT;

        Result EnsureAccess() NN_NOEXCEPT;

        Result GetLastMountFailureResult() const NN_NOEXCEPT;

        bool NeedsRebootOnAutoBoot() const NN_NOEXCEPT
        {
            return m_IsCardUpdateAppliedOnAutoBoot;
        }

        void DisableSignalEventOnRemove() NN_NOEXCEPT
        {
            m_IsSignalEventDisabledOnRemove = true;
        }

        Result ListApplicationId(int* outCount, ncm::ApplicationId* outIdList, int numList)  NN_NOEXCEPT;

    private:
        void HandleGameCardEvent() NN_NOEXCEPT;
        void HandleInserted() NN_NOEXCEPT;
        void HandleRemoved() NN_NOEXCEPT;
        Result CheckSystemUpdateNeed(bool* outValue) NN_NOEXCEPT;
        Result TryLoadAutoBoot() NN_NOEXCEPT;
        Result Load() NN_NOEXCEPT;
        Result ActivateContentManager() NN_NOEXCEPT;
        Result InactivateContentManager() NN_NOEXCEPT;
        Result RegisterContentManager() NN_NOEXCEPT;
        void UnregisterContentManager() NN_NOEXCEPT;
        Result RecordInsertedEvent(ncm::ApplicationId appId, ncm::ContentMetaDatabase* pDb) NN_NOEXCEPT;
        Result RecordGameCardInsertedEvent() NN_NOEXCEPT;
        Result RecordRemovedEvent(ncm::ApplicationId appId) NN_NOEXCEPT;
        Result RecordGameCardRemovedEvent() NN_NOEXCEPT;
        Result LoadControlData() NN_NOEXCEPT;
        Result LoadControlDataImpl(ncm::ApplicationId appId, ncm::ContentMetaDatabase* db, ncm::ContentStorage* storage) NN_NOEXCEPT;
        Result RegisterToIntegratedContentManager() NN_NOEXCEPT;
        void UnregisterFromIntegratedContentManager() NN_NOEXCEPT;
        Result RegisterToApplicationRecordDatabaseManager() NN_NOEXCEPT;
        void UnregisterFromApplicationRecordDatabaseManager() NN_NOEXCEPT;
        Result Unload() NN_NOEXCEPT;
        Result LoadGameCardInfo(const fs::GameCardHandle& handle) NN_NOEXCEPT;
        void ClearGameCardInfo() NN_NOEXCEPT;

        void EnableAutoMount() NN_NOEXCEPT;
        void DisableAutoMount() NN_NOEXCEPT;

        void StartAutoMountThread() NN_NOEXCEPT;

        void HandleMountResult(Result result) NN_NOEXCEPT;

        IntegratedContentManager* m_ContentManager;
        ApplicationRecordDatabase* m_RecordDb;
        ApplicationControlDataManager* m_ControlDataManager;
        AppletLauncher* m_AppletLauncher;
        ManualClearSystemEvent m_UpdateDetectionEvent;

        std::unique_ptr<fs::IEventNotifier> m_Notifier;
        os::SystemEventType m_CardDetectEvent;
        ManualClearSystemEvent m_SignalEvent;
        bool m_IsCardAttached;
        bool m_IsCardUpdateAppliedOnAutoBoot;
        bool m_IsSignalEventDisabledOnRemove;
        ManualClearSystemEvent m_MountErrorEvent;

        util::optional<os::ThreadType> m_Thread;
        uint8_t* m_Stack;
        size_t m_StackSize;

        mutable os::Mutex m_StateLock;
        mutable os::Mutex m_ThreadLock;

        int64_t m_EventCount;
        util::optional<ncm::ApplicationId> m_ApplicationIdForAutoBoot;

        os::Event m_StopEvent;
        sf::SharedPointer<ns::detail::IGameCardStopper> m_Stopper;

        NonRecursiveMutex m_GameCardInfoLock;
        util::optional<GameCardInfo> m_GameCardInfo;

        Result m_LastMountUnexpectedResult{ ResultSuccess() };
    };
}}}
