﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_Optional.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/ns/detail/ns_IDevelopInterface.sfdl.h>
#include <nn/ns/srv/ns_IEventDispatcher.h>
#include <nn/ns/srv/ns_IEventHandler.h>
#include <nn/pm/pm_ShellApi.h>

namespace nn { namespace ns { namespace srv {

    class IProgramLaunchObserver;
    class DevelopInterfaceServer
    {
    public:
        DevelopInterfaceServer();

        Result PrepareLaunchProgramFromHost(sf::Out<ProgramLaunchProperty> out, const nn::sf::InBuffer programPath) NN_NOEXCEPT;

        Result LaunchProgram(sf::Out<nn::os::ProcessId> out, const ProgramLaunchProperty& launchProperty, int32_t flags) NN_NOEXCEPT;

        Result LaunchApplication(sf::Out<nn::os::ProcessId> out, ncm::ApplicationId applicationId, int32_t flags) NN_NOEXCEPT;

        Result LaunchApplicationWithStorageId(
            sf::Out<nn::os::ProcessId> out,
            ncm::ApplicationId applicationId,
            int32_t flags,
            ncm::StorageId appStorageId,
            ncm::StorageId patchStorageId) NN_NOEXCEPT;

        Result GetShellEventHandle(nn::sf::Out<nn::sf::NativeHandle> out) NN_NOEXCEPT
        {
            os::NativeHandle handle;
            auto ret = GetShellEventHandle(&handle);
            out.Set(nn::sf::NativeHandle(handle, false));
            return ret;
        }

        Result GetShellEventInfo(nn::sf::Out<ShellEventInfo> out) NN_NOEXCEPT
        {
            return GetShellEventInfo(out.GetPointer());
        }

        Result TerminateProcess(os::ProcessId id) NN_NOEXCEPT;

        Result TerminateProgram(ncm::ProgramId id) NN_NOEXCEPT;

        Result TerminateApplication() NN_NOEXCEPT;

        Result IsSystemMemoryResourceLimitBoosted(nn::sf::Out<bool> pOut) NN_NOEXCEPT;

        Result GetRunningApplicationProcessId(sf::Out<nn::os::ProcessId> out) NN_NOEXCEPT;

        void SetCurrentBoostedSystemMemoryResourceLimitValue(int64_t boostSize) NN_NOEXCEPT;

    private:
        int64_t m_CurrentBoostedSystemMemoryResourceLimitValue = 0;
        os::SdkMutex m_BoostedSystemMemoryResourceLimitValueMutex;

    private:
        Result GetShellEventHandle(os::NativeHandle* pOut) NN_NOEXCEPT;
        Result GetShellEventInfo(ShellEventInfo* pOut) NN_NOEXCEPT;

    public:
        static void SetEventDispatcher(IEventDispatcher* pDispatcher) NN_NOEXCEPT
        {
            s_pEventDispatcher = pDispatcher;
        }

        void RegisterProgramLaunchObserver(IProgramLaunchObserver& programLaunchObserver) NN_NOEXCEPT
        {
            m_pProgramLaunchObserver = &programLaunchObserver;
            m_ObserverRegisteredEvent.Signal();
        }

        Result RegisterState(Event** pOut, os::ProcessId processId) NN_NOEXCEPT;

    private:
        static const int ShellEventTargetCount = 64;

        enum EventState
        {
            EventState_Null,
            EventState_Exited,
            EventState_Exception,
            EventState_DebugRunning,
            EventState_Started,
        };
        class ShellEventState: public IEventHandler
        {
        public:
            virtual void HandleEvent();

            ShellEventState() NN_NOEXCEPT
                : processId(os::ProcessId::GetInvalidId())
                , state(EventState_Null)
            {
                processId.value = 0;
            }

        public:
            os::ProcessId       processId;
            EventState          state;
            os::SystemEvent*    pClientEvent;

            void Clear() NN_NOEXCEPT
            {
                this->state = EventState_Null;
                this->GetEvent()->Clear();
            }
            void Release() NN_NOEXCEPT
            {
                Clear();

                // 0 を空きのマーカとして使っているので最後
                this->processId = os::ProcessId::GetInvalidId();
            }
        };

        ShellEventState* FindState(os::ProcessId id) NN_NOEXCEPT
        {
            for( auto& ses : m_States )
            {
                if( ses.processId == id )
                {
                    return &ses;
                }
            }

            return NULL;
        }

        void ForceFetchShellEvent() NN_NOEXCEPT;

    private:
        util::optional<ncm::ProgramId> m_DevMenuId;
        os::SystemEvent m_ClientEvent;
        ShellEventState m_States[ShellEventTargetCount];
        IProgramLaunchObserver* m_pProgramLaunchObserver;
        os::Event m_ObserverRegisteredEvent;

    private:
        static IEventDispatcher* s_pEventDispatcher;
    };

}}}

