﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/util/util_Optional.h>
#include <nn/ns/srv/ns_ApplicationControlDataManager.h>
#include <nn/ns/srv/ns_ApplicationVersionManager.h>
#include <nn/ns/srv/ns_CommitManager.h>
#include <nn/ns/srv/ns_DownloadTaskListManager.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/ns/srv/ns_TicketManager.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

namespace nn { namespace ns { namespace srv {

    class ApplicationInstallRequestList : public RequestList
    {
    public:
        ApplicationInstallRequestList(ApplicationControlDataManager* dataManager, DownloadTaskListManager* dtlManager, ApplicationVersionManager* versionManager, CommitManager* commitManager, TicketManager* ticketManager, ApplicationRecordDatabase* recordDb, ovln::SenderForOverlayType* sender) NN_NOEXCEPT :
            m_ControlDataManager(dataManager), m_LatestDownloadTaskListETag({}), m_DtlManager(dtlManager), m_IsDownloadTaskListRequested(), m_IsSystemUpdateNotificationRequiredByDownloadTaskList(), m_LatestVersionListETag({}), m_VersionManager(versionManager), m_CommitManager(commitManager), m_IsVersionListRequested(), m_TicketManager(ticketManager), m_RecordDb(recordDb), m_Sender(sender),
            m_VersionListImportedEvent(os::EventClearMode_ManualClear, false)
        {}

        virtual bool NeedsProcess(ProcessType type) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Prepare(ProcessType type) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool HasPreparedRequest() NN_NOEXCEPT NN_OVERRIDE;
        virtual void WaitAlongWith(os::SystemEvent** events, int numEvents) NN_NOEXCEPT NN_OVERRIDE;

        virtual bool HandleDone() NN_NOEXCEPT NN_OVERRIDE;
        virtual void Cleanup() NN_NOEXCEPT NN_OVERRIDE;

        void RequestDownloadTaskList(const nn::nim::ETag& eTag, bool notifiesRequiredSystemUpdate) NN_NOEXCEPT;
        void RequestVersionList(const nn::nim::ETag& eTag) NN_NOEXCEPT;
        void RequestDownloadTicket(ncm::ApplicationId id) NN_NOEXCEPT;

        os::SystemEvent* GetVersionListImportedEvent() NN_NOEXCEPT
        {
            return &m_VersionListImportedEvent;
        }

    private:
        enum class RequestType
        {
            ApplyDelta,
            NetworkInstall,
            Control,
            DownloadTaskList,
            VersionList,
            DownloadTicket,
        };
        // 実行が要求されているリクエストの属性を表すクラス
        struct RequiredTaskAttribute
        {
            RequestType type;              // リクエストの種類
            bool        isHeavy;           // 重いリクエストかどうか
            bool        canProcessOffline; // オフラインで処理可能かどうか
        };

        typedef std::pair<nim::NetworkInstallTaskId, nim::AsyncResult> ApplicationInstallRequest;
        static const int MaxRequestCount = 1;
        util::optional<ApplicationInstallRequest> m_ApplicationInstallRequestList[MaxRequestCount];

        typedef std::pair<nim::ApplyDeltaTaskId, nim::AsyncResult> ApplyDeltaRequest;
        static const int MaxApplyDeltaRequestCount = 1;
        util::optional<ApplyDeltaRequest> m_ApplyDeltaRequestList[MaxApplyDeltaRequestCount];

        typedef std::pair<ncm::ApplicationId, nim::AsyncApplicationControlInfo> ApplicationControlRequest;
        util::optional<ApplicationControlRequest> m_ApplicationControlRequest;
        ApplicationControlDataManager* m_ControlDataManager;

        util::optional<nim::AsyncData> m_DownloadTaskListRequest;
        nim::ETag m_LatestDownloadTaskListETag;
        DownloadTaskListManager* m_DtlManager;
        bool m_IsDownloadTaskListRequested;
        bool m_IsSystemUpdateNotificationRequiredByDownloadTaskList;

        util::optional<nim::AsyncData> m_VersionListRequest;
        nim::ETag m_LatestVersionListETag;
        ApplicationVersionManager* m_VersionManager;
        CommitManager* m_CommitManager;
        bool m_IsVersionListRequested;

        util::optional<std::pair<ncm::ApplicationId, ec::system::AsyncTicketDownloadStatusForPrepurchasedContents>> m_DownloadTicketRequest;
        static const int MaxDownloadTicketCount = 8;
        ncm::ApplicationId m_DownloadTicketApplicationIdList[MaxDownloadTicketCount];
        TicketManager* m_TicketManager;
        ApplicationRecordDatabase* m_RecordDb;

        ovln::SenderForOverlayType* m_Sender;

        os::SystemEvent m_VersionListImportedEvent;

        void GetRequiredTaskAttributes(int* outCount, RequiredTaskAttribute* attributes, int count) NN_NOEXCEPT;

        Result PrepareApplicationInstallRequest(nim::NetworkInstallTaskId id) NN_NOEXCEPT;
        Result PrepareApplyDeltaRequest(nim::ApplyDeltaTaskId id) NN_NOEXCEPT;
        Result PrepareApplicationControlDataRequest(nim::NetworkInstallTaskId id) NN_NOEXCEPT;
        Result ReadApplicationControlDataRequest() NN_NOEXCEPT;
        Result PrepareDownloadTaskListRequest() NN_NOEXCEPT;
        Result PrepareVersionListRequest() NN_NOEXCEPT;
        Result PrepareDownloadTicketRequest() NN_NOEXCEPT;

        void CleanupApplicationInstallRequest() NN_NOEXCEPT;
        void CleanupApplyDeltaRequest() NN_NOEXCEPT;
        void CleanupDownloadTicketRequest() NN_NOEXCEPT;

        Result HandleApplyDeltaRequestDone(util::optional<ApplyDeltaRequest>* request) NN_NOEXCEPT;
        Result HandleApplicationInstallRequestDone(util::optional<ApplicationInstallRequest>* request, bool willSuspend) NN_NOEXCEPT;
        Result HandleDownloadTaskListRequestDone() NN_NOEXCEPT;
        Result HandleVersionListRequestDone() NN_NOEXCEPT;
        Result HandleDownloadTicketRequestDone() NN_NOEXCEPT;

        void NotifyVersionListImported() NN_NOEXCEPT;

        bool HasApplicationInstallRequest(nim::NetworkInstallTaskId id) NN_NOEXCEPT;
        bool IsFullOfApplicationInstallRequest() NN_NOEXCEPT;
        bool HasApplyDeltaRequest(nim::ApplyDeltaTaskId id) NN_NOEXCEPT;
        bool IsFullOfApplyDeltaRequest() NN_NOEXCEPT;
        bool NeedsDownloadApplicationControl(const nim::NetworkInstallTaskInfo& info) NN_NOEXCEPT;
        bool NeedsProcessVersionListRequest() NN_NOEXCEPT;
        bool NeedsDownloadTicketRequest() NN_NOEXCEPT;

        Result SendContentAvailableOverlayNotification(ncm::ApplicationId id) NN_NOEXCEPT;
    };
}}}
