﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_StaticAssert.h>
#include <nn/ns/srv/detail/json/ns_Execution.h>
#include <nn/ns/ns_Result.h>

namespace nn { namespace ns { namespace srv { namespace detail { namespace json {

/** @brief JSON文字列をパースし、JSONパスと値の組をアダプタに注入します。

    @tparam AdaptorType
        JSON をパースして得られる JSON パス文字列とその値を受け取るためのクラスを指定します。
        AdaptorType では次の公開型と公開メソッドを定義する必要があります。
        - JsonPathType
            - template nn::ns::json::JsonPath<int, int> を特殊化した JSON パス文字列型
        - void Update(const JsonPathType& jsonPath, std::nullptr_t);
            - パース中に値 'null' をもつJSONパスが見つかった時に呼ばれます。
        - void Update(const JsonPathType& jsonPath, bool value);
            - パース中に bool 値をもつJSONパスが見つかった時に呼ばれます。
        - void Update(const JsonPathType& jsonPath, int64_t value);
            - パース中に std::numeric_limit<int64_t>::max 以下の整数値をもつJSONパスが見つかった時に呼ばれます。
        - void Update(const JsonPathType& jsonPath, uint64_t value);
            - パース中に std::numeric_limit<int64_t>::max より大きい整数値をもつJSONパスが見つかった時に呼ばれます。
        - void Update(const JsonPathType& jsonPath, double value);
            - パース中に浮動小数点小数値をもつJSONパスが見つかった時に呼ばれます。
        - void Update(const JsonPathType& jsonPath, const char* value, int valueLength);
            - パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。

    @tparam InputStreamType
        JSON データを入力するためのストリームクラスを指定します。
        InputStreamType では次の公開型と公開メソッドを定義する必要があります。
        - Ch
            - 必ず char を typedef してください。
        - Ch Peek() const;
            - 現在位置の文字を読み取ります。
        - Ch Take();
            - 現在位置の文字を読み取り、現在位置を次の文字に進めます。
        - size_t Tell();
            - ストリームの開始位置から現在位置までのバイト数を返します。
        - Ch* PutBegin();
            - 文字列バッファの先頭を指すポインタを返します。
        - void Put(Ch c);
            - 文字列バッファに指定した 1 文字を追加します。
        - void Flush();
            - 内容は任意です。(呼ばれることはありません)
        - size_t PutEnd(Ch* begin);
            - 文字列バッファへの文字の追加を終了します。
            - begin には PutBegin() で返したポインタが指定されます。
            - 終端文字を含む文字数を返します。
        - nn::Result GetResult() const;
            - 入力処理中に発生したエラーを返します。
            - エラーが存在しない場合は nn::ResultSuccess を返してください。

    @param[in] adaptor
        パースした値を受け取るためのアダプタを指定します。
    @param[in] stream
        入力ストリームを指定します。
        パース中に呼ばれる入力ストリームの処理を中断する場合は、 別途中止可能な入力ストリームを指定する必要があります。
    @param[in] pCancellable
        パース処理を途中で中止する可能性があれば、その中止を検知するための nn::ns::detail::Cancellable オブジェクトを指定します。
        パース処理中の特定のタイミングで中止されたかどうかが検査されます。検査の間隔は一定ではありません。

    @return 処理の結果
    @retval nn::ResultSuccess               成功
    @retval nn::ns::ResultCancelled    pCancellable により処理の中止が検出された場合

    @details
        入力ストリームで与えられる JSON 文字列をパースし、 JSON パスと値の組としてアダプタに引き渡します。
 */
template <typename AdaptorType, typename InputStreamType>
Result ImportJsonByRapidJson(
    AdaptorType& adaptor,
    InputStreamType& stream,
    const detail::json::Cancellable* pCancellable = nullptr) NN_NOEXCEPT;

}}}}} // ~namespace nn::ns::srv::detail::json

/* --------------------------------------------------------------------------------------------
    実装
 */

#include <nn/ns/srv/detail/json/ns_JsonAdaptor.h>
#include <nn/ns/srv/detail/json/ns_ResultForJson.h>

#include <limits>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/ns/srv/detail/json/ns_RapidJsonConfig.h>
#include <nn/result/result_HandlingUtility.h>

// Siglo 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif
#include <rapidjson/reader.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

namespace nn { namespace ns { namespace srv { namespace detail { namespace json {

/** @brief メモリのアロケートを禁止したアロケータ
    @details
        RapidJSON に指定するアロケータの実装。
        RapidJSON を In-situ パースのモードで使用するとメモリ確保が不要になるため、 Free 以外の内部実装が呼ばれることはありません。
 */
class NoAllocatorForRapidJson
{
public:
    static const bool kNeedFree = false;

    void* Malloc(size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        NN_ABORT(
            "[nn::ns] -----------------------------------------------\n"
            "  ABORT: Memory allocation not allowed: %zu bytes\n", size);
    }
    void* Realloc(void* originalPtr, size_t originalSize, size_t newSize) NN_NOEXCEPT
    {
        NN_UNUSED(originalPtr);
        NN_UNUSED(originalSize);
        NN_UNUSED(newSize);
        NN_ABORT(
            "[nn::ns] -----------------------------------------------\n"
            "  ABORT: Memory re-allocation not allowed: %zu -> %zu bytes\n", originalSize, newSize);
    }
    static void Free(void *ptr) NN_NOEXCEPT
    {
        // NOTE 使い捨てのアロケータのため free しない
        NN_UNUSED(ptr);
    }
};

/** @brief RapidJSON のエラーを Result に置き換える関数
 */
inline Result HandleParseResult(nne::rapidjson::ParseResult r) NN_NOEXCEPT
{
    if (r)
    {
        // エラーでないなら即終了
        NN_RESULT_SUCCESS;
    }

    NN_SDK_LOG(
        "[nn::ns] WARN: JSON Parser returned %d, at %zu\n", r.Code(), r.Offset());
    switch (r.Code())
    {
    // 空ドキュメント
    case nne::rapidjson::kParseErrorDocumentEmpty:
        return ResultJsonEmpty();
    // 根要素が複数
    case nne::rapidjson::kParseErrorDocumentRootNotSingular:
        return ResultJsonDocumentError();
    // オブジェクトなのに名前がない
    // オブジェクトなのに : がない
    // オブジェクトなのに , や } がない
    // 配列なのに , や ] がない
    // 文字列なのに " で囲まれていない
    // 特定できない文法上のエラー
    case nne::rapidjson::kParseErrorObjectMissName:
    case nne::rapidjson::kParseErrorObjectMissColon:
    case nne::rapidjson::kParseErrorObjectMissCommaOrCurlyBracket:
    case nne::rapidjson::kParseErrorArrayMissCommaOrSquareBracket:
    case nne::rapidjson::kParseErrorStringMissQuotationMark:
    case nne::rapidjson::kParseErrorUnspecificSyntaxError:
        return ResultJsonSyntaxError();
    // 不正な数値
    // \uXXXX が不正
    // サロゲートが不正
    // エスケープ文字が不正
    // 文字列が不正なエンコード
    // 数値が大きすぎる
    // 小数のフォーマットが変, 仮数部がない
    // 小数のフォーマットが変, 指数部がない
    case nne::rapidjson::kParseErrorValueInvalid:
    case nne::rapidjson::kParseErrorStringUnicodeEscapeInvalidHex:
    case nne::rapidjson::kParseErrorStringUnicodeSurrogateInvalid:
    case nne::rapidjson::kParseErrorStringEscapeInvalid:
    case nne::rapidjson::kParseErrorStringInvalidEncoding:
    case nne::rapidjson::kParseErrorNumberTooBig:
    case nne::rapidjson::kParseErrorNumberMissFraction:
    case nne::rapidjson::kParseErrorNumberMissExponent:
        return ResultJsonContentError();
    // 処理が中断された
    case nne::rapidjson::kParseErrorTermination:
        return ResultJsonParsingTerminated();
    // 未知のエラー
    default:
        NN_SDK_ASSERT(r.Code() != nne::rapidjson::kParseErrorDocumentEmpty);
        return ResultJsonEnvironmentError();
    }
}

/** @brief RapidJSON で発生するイベントのハンドラ
    @details
        RapidJSON で発生するイベントのハンドラです。
        JsonAdaptorType::JsonPath を生成し、 JsonAdaptorType オブジェクトに渡すためのアダプタの役割を果たします。
 */
template <typename JsonAdaptorType>
class EventHandlerForRapidJson
    : public CancelInjectable
{
private:
    JsonAdaptorType& m_Adaptor;
    NN_STATIC_ASSERT(
        sizeof(typename JsonAdaptorType::JsonPathType) <= 512);
    typename JsonAdaptorType::JsonPathType m_JsonPath;

public:
    EventHandlerForRapidJson(
        JsonAdaptorType& jsonAdaptor,
        const Cancellable* pCancellable) NN_NOEXCEPT
        : CancelInjectable(pCancellable)
        , m_Adaptor(jsonAdaptor)
    {
    }

    // 以下 RapidJson 向け
    bool StartObject() NN_NOEXCEPT;
    bool EndObject(nne::rapidjson::SizeType memberCount) NN_NOEXCEPT;
    bool StartArray() NN_NOEXCEPT;
    bool EndArray(nne::rapidjson::SizeType elementCount) NN_NOEXCEPT;
    bool Key(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT;
    bool Null() NN_NOEXCEPT;
    bool Bool(bool b) NN_NOEXCEPT;
    bool Int(int32_t i) NN_NOEXCEPT;
    bool Uint(uint32_t u) NN_NOEXCEPT;
    bool Int64(int64_t i) NN_NOEXCEPT;
    bool Uint64(uint64_t u) NN_NOEXCEPT;
    bool Double(double d) NN_NOEXCEPT;
    bool String(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT;
};

template <typename JsonAdaptorType, typename InputStreamType>
Result ImportJsonByRapidJson(
    JsonAdaptorType& adaptor,
    InputStreamType& stream,
    const Cancellable* pCancellable) NN_NOEXCEPT
{
    typedef nne::rapidjson::GenericReader<nne::rapidjson::UTF8<>, nne::rapidjson::UTF8<>, NoAllocatorForRapidJson> Reader;
    EventHandlerForRapidJson<JsonAdaptorType> handler(adaptor, pCancellable);
    NoAllocatorForRapidJson allocator;

    auto s =  Reader(&allocator, 1u).Parse<
        nne::rapidjson::kParseInsituFlag | nne::rapidjson::kParseStopWhenDoneFlag | nne::rapidjson::kParseValidateEncodingFlag>(stream, handler);
    auto result = HandleParseResult(s);
    if (!result.IsSuccess())
    {
        if (pCancellable && pCancellable->IsCancelled())
        {
            // キャンセルが最優先
            NN_SDK_LOG(
                "[nn::ns] INFO: JSON parsing terminated by user\n");
            NN_RESULT_THROW(nn::ns::ResultCanceled());
        }
        NN_RESULT_DO(stream.GetResult()); // IO エラーは次点
        NN_RESULT_THROW(result); // JSON パーサのエラーは最低優先度
    }
    NN_RESULT_SUCCESS;
}

template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::StartObject() NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (!m_JsonPath.TryPush(Node::Object))
    {
        return false;
    }
    m_Adaptor.Update(m_JsonPath, Node::Object, true);
    return true;
#if 0
    return true
        && !IsCancelled()
        && m_JsonPath.TryPush(Node::Object);
#endif
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::EndObject(nne::rapidjson::SizeType memberCount) NN_NOEXCEPT
{
    NN_UNUSED(memberCount);
    if (IsCancelled())
    {
        return false;
    }
    m_JsonPath.Pop();
    m_Adaptor.Update(m_JsonPath, Node::Object, false);
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::StartArray() NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (!m_JsonPath.TryPush(Node::Array))
    {
        return false;
    }
    m_Adaptor.Update(m_JsonPath, Node::Array, true);
    return true;
#if 0
    return true
        && !IsCancelled()
        && m_JsonPath.TryPush(Node::Array);
#endif
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::EndArray(nne::rapidjson::SizeType elementCount) NN_NOEXCEPT
{
    NN_UNUSED(elementCount);
    if (IsCancelled())
    {
        return false;
    }
    m_JsonPath.Pop();
    m_Adaptor.Update(m_JsonPath, Node::Array, false);

    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Key(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT
{
    NN_UNUSED(length);
    NN_UNUSED(copy);
    NN_SDK_ASSERT(strlen(str) == length, "%s: %d <-> %d\n", str, strlen(str), length);
    return true
        && !IsCancelled()
        && m_JsonPath.TryPush(Node::Key, str);
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Null() NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        // m_Path が受け入れ可能な長さの場合
        m_Adaptor.Update(m_JsonPath, nullptr);
    }
    m_JsonPath.EndAcceptValue();
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Bool(bool b) NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        m_Adaptor.Update(m_JsonPath, b);
    }
    m_JsonPath.EndAcceptValue();
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Int(int32_t i) NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        m_Adaptor.Update(m_JsonPath, static_cast<int64_t>(i));
    }
    m_JsonPath.EndAcceptValue();
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Uint(uint32_t u) NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        m_Adaptor.Update(m_JsonPath, static_cast<int64_t>(u));
    }
    m_JsonPath.EndAcceptValue();
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Int64(int64_t i) NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        m_Adaptor.Update(m_JsonPath, i);
    }
    m_JsonPath.EndAcceptValue();
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Uint64(uint64_t u) NN_NOEXCEPT
{
    if (u <= static_cast<uint64_t>(std::numeric_limits<int64_t>::max()))
    {
        // RapidJson の実装上、該当するケースはないはずだが安全のため。
        return Int64(static_cast<int64_t>(u));
    }

    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        m_Adaptor.Update(m_JsonPath, u);
    }
    m_JsonPath.EndAcceptValue();
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::Double(double d) NN_NOEXCEPT
{
    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        m_Adaptor.Update(m_JsonPath, d);
    }
    m_JsonPath.EndAcceptValue();
    return true;
}
template <typename JsonAdaptorType>
inline bool EventHandlerForRapidJson<JsonAdaptorType>::String(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT
{
    NN_UNUSED(copy);
    NN_SDK_ASSERT(strlen(str) == length, "String: %s (%d <-> %d)\n", str, strlen(str), length);
    if (IsCancelled())
    {
        return false;
    }
    if (m_JsonPath.BeginAcceptValue())
    {
        m_Adaptor.Update(m_JsonPath, str, length);
    }
    m_JsonPath.EndAcceptValue();
    return true;
}

}}}}} // ~namespace nn::ns::srv::detail::json
