﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ns/ns_Async.h>
#include <nn/ns/ns_SystemDeliveryInfo.h>

namespace nn { namespace ns {

namespace detail
{
    const char PortNameForSystemUpdate[] = "ns:su";
    class ISystemUpdateControl;
    class IAsyncValue;
    class IAsyncResult;
}

NN_DEPRECATED void InitializeForSystemUpdate() NN_NOEXCEPT;
NN_DEPRECATED void FinalizeForSystemUpdate() NN_NOEXCEPT;

/**
* @brief    バックグラウンドでダウンロードされている本体更新の状態です。
*/
enum class BackgroundNetworkUpdateState : Bit8
{
    None,       //!< 本体更新が検知されていません
    InProgress, //!< 本体更新をダウンロード中です
    Ready,      //!< 本体更新のダウンロードが完了しました
};

/**
* @brief    最新の本体更新情報です。
*/
enum class LatestSystemUpdate : Bit8
{
    UpToDate,       //!< ローカルのバージョンは最新です。
    Downloaded,     //!< 最新の本体更新がダウンロード済みです。
    NeedsDownload,  //!< 最新の本体更新をダウンロードする必要があります。
};

/**
* @brief    本体更新の進捗です。
*
* @details  トータルの容量が確定していない状態は total に 0 が入っています。
*           トータルの容量が確定していない状態でも loaded はある程度進みます。
*/
struct SystemUpdateProgress
{
    int64_t loaded; //!< ロードされた容量です。
    int64_t total;  //!< トータルの容量です。
};

typedef AsyncValue<LatestSystemUpdate> AsyncLatestSystemUpdate;

/**
* @brief    バックグラウンドでダウンロードされている本体更新の状態を取得します。
*
* @details  状態は常に変化する可能性があります。例えば Ready 状態を取得した直後に新しい
*           本体更新が通知されると、次の時点での取得では InProgress に変化しています。
*/
BackgroundNetworkUpdateState GetBackgroundNetworkUpdateState() NN_NOEXCEPT;

/**
* @brief    コンテンツ配信のために本体更新が必要な状態がシグナルされるイベントを取得します。
*
* @details  取得したイベントは、コンテンツ配信のために本体更新が必要な状態では常にシグナルされています。
*           クライアント側でイベントをクリアしてはいけません。
*/
void GetSystemUpdateNotificationEventForContentDelivery(os::SystemEvent* outValue) NN_NOEXCEPT;

class SystemUpdateControl
{
    NN_DISALLOW_COPY(SystemUpdateControl);

public:
    SystemUpdateControl() NN_NOEXCEPT;

    explicit SystemUpdateControl(sf::SharedPointer<detail::ISystemUpdateControl> interfac) NN_NOEXCEPT;

    SystemUpdateControl(SystemUpdateControl&& rvalue) NN_NOEXCEPT;

    SystemUpdateControl& operator=(SystemUpdateControl&& rvalue) NN_NOEXCEPT;

    void swap(SystemUpdateControl& other) NN_NOEXCEPT;

    /**
    * @brief    デストラクタです。
    *
    * @details  Occupy の呼び出しが成功している場合は、Relieve が呼び出されます。
    *
    */
    ~SystemUpdateControl() NN_NOEXCEPT;

    /**
    * @brief    本体更新制御を占有します。
    *
    * @details  本体更新制御に関する機能を呼び出すためには本体更新制御を占有する必要があります。
    *           この関数を呼び出すと、バックグラウンドの本体更新ダウンロードは停止します。
    *           その際ダウンロードが停止するまでこの関数は返りません。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultAlreadyOccupied   既に本体更新制御が占有されています。
    *
    * @pre
    *           - Occupy の呼び出しが成功していない
    *
    * @post
    *           - データベースファイルにメタ情報が永続化される
    */
    Result Occupy() NN_NOEXCEPT;

    /**
    * @brief    占有している本体更新制御を解放します。
    *
    * @details  バックグラウンドの本体更新ダウンロードは再開されます。
    *           カード本体更新の準備状態は全て削除されます。削除が完了するまでこの関数は返りません。
    * @pre
    *           - Occupy の呼び出しが成功している
    * @post
    *           - 占有状態が開放される
    */
    void Relieve() NN_NOEXCEPT;

    /**
    * @brief    本体更新がダウンロード済みかを取得します。
    *
    * @details  本体更新がダウンロード済みでも最新である保証はありません。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    */
    bool HasDownloaded() NN_NOEXCEPT;

    /**
    * @brief    最新の本体更新をサーバに問い合わせて、ローカルの状態と比較します。
    *
    * @details  非同期処理をリクエストして、AsyncLatestSystemUpdate を返します。
    *           処理の結果は AsyncLatestSystemUpdate から取得してください。
    *           処理が成功するためには、インターネット接続要求が受理されている必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           また、RequestDownloadLatestUpdate と同時に処理を行うことは出来ません。
    *
    * @return   処理の結果が返ります。下記以外の Result が発生した場合は、エラーコードを表示してください。
    * @retval   ResultInternetRequestNotAccepted    インターネット接続要求が受理されていません。
    * @retval   ResultOutOfMaxRunningTask           同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultCanceled                      （AsyncLatestSystemUpdate から返される Result）処理がキャンセルされました。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    */
    Result RequestCheckLatestUpdate(AsyncLatestSystemUpdate* outValue) NN_NOEXCEPT;

    /**
    * @brief    最新の本体更新をダウンロードします。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インターネット接続要求が受理されている必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           また、RequestCheckLatestUpdate と同時に処理を行うことは出来ません。
    *
    * @return   処理の結果が返ります。下記以外の Result が発生した場合は、エラーコードを表示してください。
    * @retval   ResultInternetRequestNotAccepted    インターネット接続要求が受理されていません。
    * @retval   ResultOutOfMaxRunningTask           同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultAlreadyUpToDate               （AsyncResult から返される Result）ローカルのバージョンが最新です。
    * @retval   ResultCanceled                      （AsyncResult から返される Result）処理がキャンセルされました。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *
    * @post     - HasDownloaded() == true
    */
    Result RequestDownloadLatestUpdate(AsyncResult* outValue) NN_NOEXCEPT;

    /**
    * @brief    本体更新のダウンロード進捗を取得します。
    *
    * @details  この関数は他の関数呼び出しとは異なるスレッドから呼び出すことが可能です。
    *           必要なトータルの容量が不明な時点では、Progress::total は 0 が返ります。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    */
    SystemUpdateProgress GetDownloadProgress() NN_NOEXCEPT;


    /**
    * @brief    ダウンロード済みの本体更新から EULA データのサイズを取得します。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - HasDownloaded() == true
    *           - strlen(eulaDataPath) <= 255
    *
    * @return   処理の結果が返ります。
    * @retval   ResultEulaSystemDataNotFound    ダウンロードされた本体更新に EULA システムデータが含まれていません。
    * @retval   ResultEulaDataNotPathFound      EULA システムデータに指定のパスが見つかりません。
    */
    Result GetDownloadedEulaDataSize(size_t* outValue, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief    ダウンロード済みの本体更新から EULA データを取得します。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - HasDownloaded() == true
    *           - strlen(eulaDataPath) <= 255
    *
    * @return   処理の結果が返ります。
    * @retval   ResultEulaSystemDataNotFound    ダウンロードされた本体更新に EULA システムデータが含まれていません。
    * @retval   ResultEulaDataNotPathFound      EULA システムデータに指定のパスが見つかりません。
    * @retval   ResultBufferNotEnough           与えられたバッファよりも EULA データが大きいです。
    */
    Result GetDownloadedEulaData(size_t* outValue, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief    ダウンロード済みの本体更新を適用します。
    *
    * @details  適用が完了するまでこの関数は返りません。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - HasDownloaded() == true
    */
    void ApplyDownloadedUpdate() NN_NOEXCEPT;

    /**
    * @brief    カード本体更新の事前準備を行います。
    *
    * @details  ダウンロード中の本体更新は全て削除されます。
    *           この関数が成功するためには、カードが読み込み可能である必要があります。
    *           事前準備が完了するまでこの関数は返りません。
    *
    *           与えられたバッファはカード本体更新のファイルシステム読み書きに利用されます。
    *           バッファが大きい方がカード本体更新は速く完了します。
    *           与えられたバッファは SystemUpdateControl が破棄されるか Relieve() が呼び出される
    *           までユーザ側で参照することができなくなります。
    *           バッファのアドレスとサイズは os::MemoryPageSize の整数倍である必要があります。
    *
    * @return   処理の結果が返ります。カードの読込に失敗したら下位の Result が返ります。
    * @retval   ResultAlreadyUpToDate         ローカルのバージョンが最新です。
    *           ResultCardUpdateAlreadySetup  すでに Setup を呼んでいる状態です。
    *
    * @pre
    *           - buffer % os::MemoryPageSize == 0
    *           - bufferSize % os::MemoryPageSize == 0
    *           - Occupy の呼び出しが成功している
    *
    * @post     - HasPrepareCardUpdate() == true
    */
    Result SetupCardUpdate(void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /**
    * @brief    カード本体更新を本体更新準備領域に書き込みます。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           AsyncResult に対して Cancel を呼んだ場合、再度カード本体更新を実行するためには、
    *           SystemUpdateControl を破棄するか、Release() を呼び出したのちに、
    *           SystemUpdateControl の再取得を行う必要があります。
    *           なお、正しくキャンセルされた場合は AsyncResult からは ns::ResultCanceled が返ります。
    *
    * @return   処理の結果が返ります。カードの読込に失敗したら下位の Result が返ります。
    * @retval   ResultCardUpdateNotSetup   カード本体更新の事前準備が完了していません。
    * @retval   ResultOutOfMaxRunningTask  同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultPrepareCardUpdateAlreadyRequested  過去に Request したことがあるため、PrepareCardUpdate を実行できません。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupCardUpdate の呼び出しが成功している
    *
    * @post     - HasPrepareCardUpdate() == true
    */
    Result RequestPrepareCardUpdate(AsyncResult* outValue) NN_NOEXCEPT;

    /**
    * @brief    カード本体更新の準備領域書き込み進捗を取得します。
    *
    * @details  必要なトータルの容量が不明な時点では、Progress::total は 0 が返ります。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupCardUpdate の呼び出しが成功している
    */
    SystemUpdateProgress GetPrepareCardUpdateProgress() NN_NOEXCEPT;

    /**
    * @brief    カード本体更新の準備領域書き込みが完了したか確認します。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupCardUpdate の呼び出しが成功している
    */
    bool HasPreparedCardUpdate() NN_NOEXCEPT;

    /**
    * @brief    本体更新準備領域に書き込まれたカード本体更新から EULA データのサイズを取得します。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupCardUpdate の呼び出しが成功している
    *           - HasPrepareCardUpdate() == true
    *           - strlen(eulaDataPath) <= 255
    *
    * @return   処理の結果が返ります。
    * @retval   ResultEulaSystemDataNotFound    ダウンロードされた本体更新に EULA システムデータが含まれていません。
    * @retval   ResultEulaDataNotPathFound      EULA システムデータに指定のパスが見つかりません。
    */
    Result GetPreparedCardUpdateEulaDataSize(size_t* outValue, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief    本体更新準備領域に書き込まれたカード本体更新から EULA データを取得します。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupCardUpdate の呼び出しが成功している
    *           - HasPrepareCardUpdate() == true
    *           - strlen(eulaDataPath) <= 255
    *
    * @return   処理の結果が返ります。
    * @retval   ResultEulaSystemDataNotFound    ダウンロードされた本体更新に EULA システムデータが含まれていません。
    * @retval   ResultEulaDataNotPathFound      EULA システムデータに指定のパスが見つかりません。
    * @retval   ResultBufferNotEnough           与えられたバッファよりも EULA データが大きいです。
    */
    Result GetPreparedCardUpdateEulaData(size_t* outValue, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief    本体更新準備領域に書き込まれたカード本体更新を適用します。
    *
    * @details  適用が完了するまでこの関数は返りません。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupCardUpdate の呼び出しが成功している
    *           - HasPrepareCardUpdate() == true
    */
    void ApplyCardUpdate() NN_NOEXCEPT;

    /**
    * @brief    本体更新準備領域に書き込まれたカード本体更新を適用します。
    *
    * @details  適用が完了する、もしくはエラーが発生するまでこの関数は返りません。
    *           本関数は自動テスト検証のためのデバッグ版インタフェースです。
    *           Result のエラーハンドリングを含めた正規処理には ApplyCardUpdate を使用してください。
    *           ※本ドキュメントのResult値説明には詳細な内部エラーコードは含まれていません。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupCardUpdate の呼び出しが成功している
    *           - HasPrepareCardUpdate() == true
    *
    * @return   処理の結果が返ります。
    * @retval   ResultSuccess                   成功しました。
    * @retval   ResultCardUpdateNotSetup        SetupCardUpdate の呼び出しに成功していません。
    * @retval   ResultCardUpdateNotPrepared     RequestPrepareCardUpdate が成功していません。
    */
    Result ApplyCardUpdateForDebug() NN_NOEXCEPT;

    /**
    * @brief    システムアップデーターのリソースデータを参照する疑似カード本体更新の事前準備を行います。
    *
    * @details  本関数はシステムアップデーター以外からは実行しないでください。
    *           SetupCardUpdate() の代わりに実行し、以降の処理はカード本体更新と同様に行います。
    *           ダウンロード中の本体更新は全て削除されます。
    *           事前準備が完了するまでこの関数は返りません。
    *           この関数は製品環境では必ず失敗します。
    *
    *           与えられたバッファはシステムアップデーター本体更新のファイルシステム読み書きに利用されます。
    *           バッファが大きい方がシステムアップデーター本体更新は速く完了します。
    *           与えられたバッファは SystemUpdateControl が破棄されるか Relieve() が呼び出される
    *           までユーザ側で参照することができなくなります。
    *           バッファのアドレスとサイズは os::MemoryPageSize の整数倍である必要があります。
    *
    * @return   処理の結果が返ります。システムアップデーターのリソースデータの読込に失敗したら下位の Result が返ります。
    * @retval   ResultAlreadyUpToDate         ローカルのバージョンが最新です。
    *           ResultCardUpdateAlreadySetup  すでに Setup を呼んでいる状態です。
    *
    * @pre
    *           - buffer % os::MemoryPageSize == 0
    *           - bufferSize % os::MemoryPageSize == 0
    *           - Occupy の呼び出しが成功している
    *
    * @post     - HasPrepareCardUpdate() == true
    */
    Result SetupCardUpdateViaSystemUpdater(void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /**
    * @brief    本体更新がローカルコンテンツ配信によって受信済みかを取得します。
    *
    * @details  本 API は lcs から呼び出されますので、 Ocean は呼び出さないでください。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    */
    bool HasReceived() NN_NOEXCEPT;

    /**
    * @brief    ローカルコンテンツ配信を用いた本体更新の事前準備を行います。
    *
    * @details  本 API は lcs から呼び出されますので、 Ocean は呼び出さないでください。
    *           ダウンロード中の本体更新は全て削除されます。
    *           事前準備が完了するまでこの関数は返りません。
    *
    * @return   処理の結果が返ります。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *
    */
    Result SetupToReceiveSystemUpdate() NN_NOEXCEPT;

    /**
    * @brief    本体更新をローカルコンテンツ配信によって受信します。
    *
    * @details  本 API は lcs から呼び出されますので、 Ocean は呼び出さないでください。
    *           非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           AsyncResult が破棄されるとタスクも破棄されるので、適用を終えるまで AsyncResult は保持し続けてください。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultOutOfMaxRunningTask           同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultAlreadyUpToDate               （AsyncResult から返される Result）ローカルのバージョンが最新です。
    * @retval   ResultCanceled                      （AsyncResult から返される Result）処理がキャンセルされました。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - SetupToReceiveSystemUpdate の呼び出しが成功している
    *
    * @post     - HasReceived() == true
    */
    Result RequestReceiveSystemUpdate(AsyncResult* outValue, uint32_t ipv4, uint16_t port, const SystemDeliveryInfo& info) NN_NOEXCEPT;

    /**
    * @brief    本体更新の受信進捗を取得します。
    *
    * @details  本 API は lcs から呼び出されますので、 Ocean は呼び出さないでください。
    *           この関数は他の関数呼び出しとは異なるスレッドから呼び出すことが可能です。
    *           必要なトータルの容量が不明な時点では、Progress::total は 0 が返ります。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    */
    SystemUpdateProgress GetReceiveProgress() NN_NOEXCEPT;


    /**
    * @brief    受信済みの本体更新から EULA データのサイズを取得します。
    *
    * @details  本 API は lcs から呼び出されますので、 Ocean は呼び出さないでください。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - HasReceived() == true
    *           - strlen(eulaDataPath) <= 255
    *
    * @return   処理の結果が返ります。
    * @retval   ResultEulaSystemDataNotFound    ダウンロードされた本体更新に EULA システムデータが含まれていません。
    * @retval   ResultEulaDataNotPathFound      EULA システムデータに指定のパスが見つかりません。
    */
    Result GetReceivedEulaDataSize(size_t* outValue, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief    受信済みの本体更新から EULA データを取得します。
    *
    * @details  本 API は lcs から呼び出されますので、 Ocean は呼び出さないでください。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - HasReceived() == true
    *           - strlen(eulaDataPath) <= 255
    *
    * @return   処理の結果が返ります。
    * @retval   ResultEulaSystemDataNotFound    ダウンロードされた本体更新に EULA システムデータが含まれていません。
    * @retval   ResultEulaDataNotPathFound      EULA システムデータに指定のパスが見つかりません。
    * @retval   ResultBufferNotEnough           与えられたバッファよりも EULA データが大きいです。
    */
    Result GetReceivedEulaData(size_t* outValue, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief    受信済みの本体更新を適用します。
    *
    * @details  本 API は lcs から呼び出されますので、 Ocean は呼び出さないでください。
    *           適用が完了するまでこの関数は返りません。
    *
    * @pre
    *           - Occupy の呼び出しが成功している
    *           - HasReceived() == true
    */
    void ApplyReceivedUpdate() NN_NOEXCEPT;

private:
    template <typename AsyncT, typename IAsyncT, typename RequestFuncT>
    Result RequestAsyncValue(AsyncT* outValue, RequestFuncT func) NN_NOEXCEPT;

    sf::SharedPointer<detail::ISystemUpdateControl> m_Interface;
};

}}

