﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace ns {

    /**
    * @brief   ユーザーが同意の元、次回システム起動時に SD カードを使える状態にします。
    *
    * @details 本 API を呼び出すと、 SD カード内の管理情報を削除します。
    *          SD カードは管理情報を構築した本体以外で利用することはできません。
    *          他の本体で利用したい場合には本 API を呼び出し、管理情報を削除する必要があります。
    *
    *          SD カード内への管理情報の構築はシステム起動時にSD カード内に管理情報がない場合に行われます。
    *          管理情報の構築はシステムの起動時にのみ行われますので、SD カードを使える状態にするためには、
    *          本 API を呼び出した後にシステムを再起動する必要があります。
    *
    * @return 処理の結果が返ります。 Result は実際に処理を行っている他のモジュールの Result が返ってきます。
    *         Result が Failure の場合は想定外の SD カードが想定外の状態になっています。
    *
    * @pre
    *          - SD カードが現在利用中ではない
    *          - SD カードのファイルシステムが正常である
    * @post
    *          - SD カードから管理情報が削除されます
    */
    Result CleanupSdCard() NN_NOEXCEPT;

    /**
    * @brief   システムが最後に SD カードの挿抜処理を行ったときの状態を確認します。
    *
    * @details システムがSD カードの挿入に関する処理は、起動時と起動後の SD カードの挿抜時に行われます。
    *          本 API では、その処理の結果を返します。
    *
    * @return  SD カードの挿入処理の結果が返ります。Success のときは正常に使える状態です。
    * @retval  ResultSdCardNoOwnership         他の本体で初期化された SD カードが挿入されました。 SD カードを利用するためには nn::ns::CleanupSdCard() を呼ぶ必要があります。
    * @retval  ResultSdCardNeedsSystemUpdate   exfat でかつ、 exfat ドライバが必要です。 SD カードを利用するためには本体更新が必要です。
    * @retval  ResultSdCardFileSystemCorrupted ファイルシステムが壊れています。 SD カードを利用するためにはフォーマットが必要です。
    * @retval  ResultSdCardDatabaseCorrupted   データベースが壊れています。 SD カードを利用するためには nn::ns::CleanupSdCard() を呼ぶ必要があります。
    * @retval  ResultSdCardNotMounted          起動後に SD カードが挿入されました。 SD カードを利用するためには再起動が必要です。
    * @retval  ResultSdCardNotInserted         SD カードが挿入されていません。 SD カードを利用するための処理は不要です。
    */
    Result CheckSdCardMountStatus() NN_NOEXCEPT;

    /**
    * @brief    SD カードが挿抜されたことを検知するためのイベントを取得します。
    *
    * @return   クリアモードが nn::os::EventClearMode_ManualClear のイベントが返ります。
    *
    */
    void GetSdCardMountStatusChangedEvent(os::SystemEvent* pOutEvent) NN_NOEXCEPT;

    /**
    * @brief   SD カードの挿抜処理の際、想定外のエラーが発生した時の Result を取得します。
    *
    * @details 一番最後に発生した想定外の Result を取得します。
    *          もし、一度も想定外のエラーが発生していない場合、 ResultSuccess が返ります。
    *
    * @return  SD カード挿抜処理における想定外の Result
    */
    Result GetLastSdCardMountUnexpectedResult() NN_NOEXCEPT;

    /**
    * @brief    SD カードをフォーマットします
    *
    * @details  本 API を呼び出すと、SD カードが利用できなくなり、ダウンロードタスクが全てキャンセルされます。
    *           パッチ間差分適用タスクは、SD カード向けのタスクがすべてキャンセルされます。
    *           本 API を呼び出した後は再起動してください。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultSdCardFormatUnexpected    想定外のエラーが発生しました。
    */
    Result FormatSdCard() NN_NOEXCEPT;

    /**
    * @brief    SD カードをフォーマットする際にシステムアップデートが必要かを確認します
    *
    * @return   処理の結果が返ります。
    * @retval   ResultSdCardFormatUnexpected    想定外のエラーが発生しました。
    */
    Result NeedsSystemUpdateToFormatSdCard(bool* outValue) NN_NOEXCEPT;

    /**
    * @brief   ResultSdCardFormatUnexpected が発生した時の Result を取得します。
    *
    * @details 一番最後に ResultSdCardFormatUnexpected  が発生したときの詳細な Result を取得します。
    *          もし、一度も ResultSdCardFormatUnexpected が発生していない場合、 ResultSuccess が返ります。
    *
    * @return  SD カードフォーマット処理における想定外の Result
    */
    Result GetLastSdCardFormatUnexpectedResult() NN_NOEXCEPT;
}}  // namespace ns
