﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/nim/nim_ETag.h>
#include <nn/time/time_Types.h>

namespace nn { namespace ns {
    struct NotificationInfo {
        struct WaitingPolicy
        {
            enum class WaitingPolicyType : Bit8 {
                RandomWaiting,
            };

            WaitingPolicyType type;
            Bit8 reserved[3];
            int waitingLimit;
        };

        struct NetworkUpdateInfo
        {
            Bit64       titleId;
            uint32_t    titleVersion;
            Bit8        reserved0[4];
            Bit64       requiredTitleId;
            uint32_t    requiredVersion;
            Bit8        reserved1[4];
        };

        struct DownloadTaskListInfo
        {
        };

        struct VersionListInfo
        {
            WaitingPolicy autoUpdateWaitingPolicy;
        };

        struct ETicketAvailableInfo
        {
            uint32_t accountId;
            Bit8 reserved0[4];
            uint64_t titleId;
            Bit8 reserved1[16];
        };

        struct SendRightsUsageStatusRequestInfo
        {
        };

        struct SyncELicenseRequestInfo
        {
            account::NintendoAccountId accountIds[account::UserCountMax];
            int accountCount;
            bool syncAll;
            Bit8 reserved[3];
        };

        enum class NotificationType : Bit8 {
            SystemUpdate,
            DownloadTaskList,
            VersionList,
            ETicketAvailable,
            SendRightsUsageStatusRequest,
            SyncELicenseRequest,
            NumberOfNotificationType,
        };


        nim::ETag   eTag;
        NotificationType type;
        Bit8 reserved[7];

        time::PosixTime time;
        union {
            NetworkUpdateInfo nupInfo;
            DownloadTaskListInfo dtlInfo;
            VersionListInfo versionListInfo;
            ETicketAvailableInfo eTicketAvailableInfo;
            SendRightsUsageStatusRequestInfo sendRightsUsageStatusInfo;
            SyncELicenseRequestInfo syncELicenseRequestInfo;
        };

        WaitingPolicy waitingPolicy;
    };

    inline bool operator ==(const NotificationInfo::WaitingPolicy& lhs, const NotificationInfo::WaitingPolicy& rhs) NN_NOEXCEPT
    {
        return lhs.type == rhs.type && lhs.waitingLimit == rhs.waitingLimit;
    }

    struct AsyncTask
    {
        enum class AsyncTaskType
        {
            SystemUpdate,
            DownloadTaskList,
            VersionList,
            AutoUpdate,
            ETicketAvailable,
            SendRightsUsageStatusRequest,
            SyncELicenseRequest,

            NumberOfAsyncTaskType,
        };

        AsyncTaskType taskType;
        NotificationInfo info;
        int64_t waitingLimit;
        int64_t registeredTime;
        int64_t lastUpdatedTime;
    };

    bool IsNotificationSetupCompleted() NN_NOEXCEPT;

    int64_t GetLastNotificationInfoCount() NN_NOEXCEPT;

    int ListLastNotificationInfo(NotificationInfo outList[], int count) NN_NOEXCEPT;
    int ListNotificationTask(AsyncTask outList[], int count) NN_NOEXCEPT;
}}  // namespace nn::ns
