﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ns/ns_Async.h>

namespace nn { namespace ns {
    typedef AsyncValue<int> AsyncGameCardGoldPoint;

    /**
    * @brief    ゲームカードの登録状況を取得します。未登録の場合に ResultSuccess が返ります。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *
    *           この関数で同時にリクエストできる非同期処理は最大１つで、このヘッダに
    *           記述されている Request～ 系関数と同時非同期処理可能数を共有しています。
    *           別の Request 関数で非同期処理を開始するためには、取得した Async～
    *           オブジェクトを破棄する必要があります。
    *
    *           現時点で、アプリケーション ID とゲームカードの一貫性はチェックしていません。
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
    * @retval   ResultOutOfMaxRunningTask           同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultGameCardAlreadyRegistered     ゲームカードがすでに登録されている。
    * @retval   ResultCanceled                      （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestCheckGameCardRegistration(AsyncResult* outValue, ncm::ApplicationId appId) NN_NOEXCEPT;

    /**
    * @brief    ユーザーアカウント・アプリケーション ID を指定してゲームカードに対する GoldPoint を取得します。
    *
    * @details  ユーザーアカウントにニンテンドーアカウントが紐付けられている必要があります。
    *
    *           非同期処理をリクエストして、AsyncGameCardGoldPoint を返します。
    *           処理の結果は AsyncGameCardGoldPoint から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *
    *           この関数で同時にリクエストできる非同期処理は最大１つで、このヘッダに
    *           記述されている Request～ 系関数と同時非同期処理可能数を共有しています。
    *           別の Request 関数で非同期処理を開始するためには、取得した Async～
    *           オブジェクトを破棄する必要があります。
    *
    *           現時点で、アプリケーション ID とゲームカードの一貫性はチェックしていません。
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
    * @retval   ResultOutOfMaxRunningTask           同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultCanceled                      （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestGameCardRegistrationGoldPoint(AsyncGameCardGoldPoint* outValue, const account::Uid& uid, ncm::ApplicationId appId) NN_NOEXCEPT;

    /**
    * @brief    ユーザーアカウント・アプリケーション ID を指定してゲームカードを登録します。
    *
    * @details  ユーザーアカウントにニンテンドーアカウントが紐付けられている必要があります。
    *
    *           非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *
    *           この関数で同時にリクエストできる非同期処理は最大１つで、このヘッダに
    *           記述されている Request～ 系関数と同時非同期処理可能数を共有しています。
    *           別の Request 関数で非同期処理を開始するためには、取得した Async～
    *           オブジェクトを破棄する必要があります。
    *
    *           現時点で、アプリケーション ID とゲームカードの一貫性はチェックしていません。
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
    * @retval   ResultOutOfMaxRunningTask           同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultCanceled                      （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestRegisterGameCard(AsyncResult* outValue, const account::Uid& uid, ncm::ApplicationId appId, int goldPoint) NN_NOEXCEPT;
}}
