﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationVerificationTypes.h>
#include <nn/ns/ns_Async.h>
#include <nn/ns/ns_AsyncProgress.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace ns {

typedef ProgressAsyncResult<AsyncVerifyApplicationProgress> AsyncVerifyApplicationResult;
typedef ProgressAsyncResult<AsyncVerifyAddOnContentsProgress> AsyncVerifyAddOnContentsResult;


/**
* @brief    アプリケーションが保持するコンテンツの破損チェックを非同期処理で要求します。
*
* @details  非同期処理をリクエストして AsyncResult を返します。
*           処理の結果は AsyncResult から取得してください。
*           処理が成功すると、バックグラウンドでアプリケーションの破損チェックが始まります。
*           非同期処理の内部で利用するワーキングバッファをアプリケーションから渡して貰う必要があります。
*           ワーキングバッファはアドレス及びサイズが 4KB にアラインされている必要があります。
*
*           この関数で同時にリクエストできる非同期処理は最大１つです。
*           また、下記の関数と最大非同期処理リクエスト可能数を共有しています。
*           - RequestUpdateApplication
*           - RequestApplicationUpdateInfo
*           - RequestDownloadApplicationControlData
*           - RequestVerifyApplication
*           - RequestVerifyAddOnContentsRights
*
* @return   処理の結果が返ります。
* @retval   ResultOutOfMaxRunningTask           同時に発行できるリクエスト数が最大に達しています。
* @retval   ResultCanceled                      （AsyncVerifyApplicationResult から返される Result ）処理がキャンセルされました。
* @retval   ResultVerificationFailed            （AsyncVerifyApplicationResult から返される Result ）コンテンツの検証に失敗しました。 ResultContentCorrupted や ResultVerifyContentUnexpected を含みます。
* @retval   ResultContentCorrupted              （ResultVerificationFailed に含まれる Result ）コンテンツが破損しています。
* @retval   ResultVerifyContentUnexpected       （ResultVerificationFailed に含まれる Result ）コンテンツの検証に失敗しました。nn::ns::AsyncVerifyApplicationResult::GetDetailResult() で詳細な Result が取得できます。
* @retval   ResultSdCardDataCorrupted           （AsyncVerifyApplicationResult から返される Result ）SD カードのデータが破損しています。
*/
Result RequestVerifyApplication(AsyncVerifyApplicationResult* outValue, ncm::ApplicationId id, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

Result RequestVerifyApplication(AsyncVerifyApplicationResult* outValue, ncm::ApplicationId id, VerifyContentFlag flags, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
 * @brief   アプリケーションを破壊します。
 *
 * @details 本 API を呼び出した後、そのアプリケーションを起動させると MaybeCorrupted フラグが立ちます。
 *          GameCard に存在するコンテンツは対象外です。
 *          storageId には破壊するコンテンツがインストールされているストレージを指定してください。
 *          有効なストレージすべてを対象としたい場合は、 nn::ncm::StorageId::Any を指定してください。
 *
 * @return  処理の結果が返ります。
 * @retval  ResultApplicationRecordNotFound     アプリケーション記録が存在しません
 * @retval  ResultApplicationContentNotFound    flags に指定したコンテンツが見つかりません
 * @retval  ResultUnsupportedStorage            有効なストレージが指定されていません
 */
Result CorruptApplicationForDebug(ncm::ApplicationId id, CorruptContentFlag flags, ncm::StorageId storageId) NN_NOEXCEPT;

/**
 * @brief   コンテンツを破壊します。
 *
 * @details 破壊するコンテンツを指定して破壊します。
 *          GameCard に存在するコンテンツは対象外です。
 *          storageId には破壊するコンテンツがインストールされているストレージを指定してください。
 *          有効なストレージすべてを対象としたい場合は、 nn::ncm::StorageId::Any を指定してください。
 *
 * @return  処理の結果が返ります。
 * @retval  ResultContentMetaNotFound           指定したコンテンツが見つかりませんでした。
 */
Result CorruptContentForDebug(const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT;


/*
* @brief    追加コンテンツの権限チェックを非同期処理で要求します。
*
* @details  非同期処理をリクエストして AsyncResult を返します。
*           処理の結果は AsyncResult から取得してください。
*           処理が成功すると、バックグラウンドで追加コンテンツの権利チェックが始まります。
*
*           この関数で同時にリクエストできる非同期処理は最大１つです。
*           また、下記の関数と最大非同期処理リクエスト可能数を共有しています。
*           - RequestUpdateApplication
*           - RequestApplicationUpdateInfo
*           - RequestDownloadApplicationControlData
*           - RequestVerifyApplication
*           - RequestVerifyAddOnContentsRights
*
* @return   処理の結果が返ります。
* @retval   ResultOutOfMaxRunningTask                 同時に発行できるリクエスト数が最大に達しています。
* @retval   ResultCanceled                           （AsyncVerifyAddOnContentsResult から返される Result ）処理がキャンセルされました。
* @retval   ResultVerificationFailed                 （AsyncVerifyAddOnContentsResult から返される Result ）コンテンツの検証に失敗しました。 ResultContentCorrupted や ResultVerifyContentUnexpected を含みます。
* @retval   ResultContentCorrupted                   （ResultVerificationFailed に含まれる Result ）コンテンツが破損しています。
* @retval   ResultVerifyContentUnexpected            （ResultVerificationFailed に含まれる Result ）コンテンツの検証に失敗しました。nn::ns::AsyncVerifyApplicationResult::GetDetailResult() で詳細な Result が取得できます。
* @retval   ResultSdCardDataCorrupted                （AsyncVerifyAddOnContentsResult から返される Result ）SD カードのデータが破損しています。
* @retval   ResultApplicationRecordNotFound           (AsyncVerifyAddOnContentsResult から返される Result ）アプリケーション記録が存在しません
* @retval   ResultApplicationTicketNotFound           (AsyncVerifyAddOnContentsResult から返される Result ）チケットが存在しません
* @retval   ResultInactiveNintendoAccount             (AsyncVerifyAddOnContentsResult から返される Result ）チケットに対応する NA が無効です
* @retval   ResultRightsOwnerVirtualAccountNotFound   (AsyncVerifyAddOnContentsResult から返される Result ）チケットに対応する VA が存在しません
*/

Result RequestVerifyAddOnContentsRights(AsyncVerifyAddOnContentsResult* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

}}  // namespace nn::ns
