﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Result.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationManagerInitializeApi.h>
#include <nn/ns/ns_Async.h>
#include <nn/ns/ns_GameCardApi.h>
#include <nn/ns/ns_SdCardApi.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace ns {

/**
* @brief    アプリケーションに対して発生するイベントです。
*/
enum class ApplicationEvent : Bit8
{
    Launched,           //!< アプリケーションが起動されました。
    LocalInstalled,     //!< アプリケーションに関連するコンテンツのローカルインストールが完了しました。
    DownloadStarted,    //!< アプリケーションに関連するコンテンツのダウンロードが開始されました。
    GameCardInserted,   //!< アプリケーションが格納されているゲームカードが挿入されました。
    Touched,            //!< アプリケーション記録が明示的に更新されました。
    GameCardRemoved,    //!< ゲームカードが抜去されました。
    AttributeUpdated,   //!< アプリケーション記録の属性が更新されました
    DownloadTaskCommitted,           //!< ダウンロードタスクがコミットされました
    DownloadTaskCommittedPartially,  //!< ダウンロードタスクの一部のみがコミットされました
    ReceiveTaskCommitted,            //!< ローカル配信タスクがコミットされました
    ApplyDeltaTaskCommitted,         //!< パッチ間差分適用タスクがコミットされました
    Deleted,            //!< アプリケーションの整理が行われました。
    PropertyUpdated,    //!< アプリケーション記録データベースのプロパティが更新されました。
    ContentAvailable,   //!< コンテンツが利用可能になりました。
};

/**
* @brief    アプリケーション記録です。
*/
struct ApplicationRecord
{
    ncm::ApplicationId      id;             //!< アプリケーションの ID です。
    ApplicationEvent        lastEvent;      //!< アプリケーションの最後のイベントです。
    Bit8                    attributes;     //!< システム内部で使用する属性です。
    Bit8                    reserved[6];
    int64_t                 lastUpdated;    //!< アプリケーションの最後のイベントが発生したカウントです。
};

/**
* @brief    更新された順にアプリケーション記録を列挙します。
*
* @details  アプリケーション記録は、以下のイベントで追加・更新され、以後削除されるまで永続化されます。
*           @li アプリケーションが起動した時
*           @li アプリケーションが終了した時
*           @li アプリケーションが格納されている外部メディア（現状 ROM カード）が挿入された時
*           @li アプリケーション本体・追加コンテンツ・パッチのいずれかのダウンロードが開始された時
*           当該イベントおよびその発生タイムスタンプは ApplicationRecord::lastEvent, lastUpdated
*           で取得できます。
*
*           中断中のアプリケーションは常に先頭にリストアップされます。その際、lastEvent は Running で
*           lastUpdated は無効値です。
*
*           アプリケーション記録は内部的にアプリケーション本体・追加コンテンツ・パッチのインストール記録及び
*           バージョンを記録しており、各種アプリケーション記録を利用する API 内部で利用されます。
*
*           アプリケーション記録は、Application ID 単位で追加・更新され、ID の重複はありません。
*
*           ListApplicationRecord で列挙されるアプリケーション記録は lastUpdated が新しい順に
*           ソートされています。
*
* @return   列挙されたアプリケーション記録の数が返ります。
*
* @pre
*           - outValue には count 分以上の配列が確保されている。
*
* @post
*           - outValue にリストアップされたアプリケーションが書き込まれている。
*/
int ListApplicationRecord(ApplicationRecord outValue[], int count, int offset) NN_NOEXCEPT;

/**
* @brief    アプリケーション記録を明示的に更新します。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*/
Result TouchApplication(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    アプリケーション記録と比較可能なカウントを取得します。
*
* @details  アプリケーション記録と同列に表示したいコンテンツがある場合に当該コンテンツに
*           割り当てるカウントをこの API で取得してください。
*/
int64_t GenerateApplicationRecordCount() NN_NOEXCEPT;

/**
* @brief    アプリケーション記録の変化を検知するシステムイベントを取得します。
*
* @details  取得したシステムイベントは、アプリケーション記録の追加・削除・更新が発生した時にシグナルされます。
*           SystemEvent のクリアモードは EventClearMode_AutoClear です。
*
* @pre
*           - outValue は未初期化の SystemEvent
*
* @post
*           - outValue に初期化済みの SystemEvent が格納される
*/
void GetApplicationRecordUpdateSystemEvent(os::SystemEvent* outValue) NN_NOEXCEPT;

/**
* @brief    指定したアプリケーションを大切なアプリケーションとしておすすめ削除の対象から除外します。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*/
Result DisableApplicationAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    指定したアプリケーションを通常のアプリケーションとしておすすめ削除の対象にします。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*/
Result EnableApplicationAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    アプリケーションがアップデートを必要としているかを判定します。
*
* @detail   アップデートを必要としている場合、 outValue に更新理由が入ります。
*           一方で、アップデートを必要としていない場合、outValue には util::nullopt が入ります。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*/
Result IsApplicationUpdateRequested(util::optional<Result>* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    アプリケーションがアップデートを完了したことを通知します。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*/
Result WithdrawApplicationUpdateRequest(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    追加コンテンツのクリーンナップが終了したことを通知します。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*/
Result WithdrawCleanupAddOnContentsWithNoRightsRecommendation(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    アプリケーションの記録が存在するかを確認します。
*
* @detail   当該 API は ns:web もしくは ns:am2 権限で利用することが可能です。
* @return   アプリケーションの記録が存在すれば True を返します。
*/
bool HasApplicationRecord(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    指定したアプリケーションをプリインストール権利がある状態にします。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*/
Result SetPreInstalledApplication(ncm::ApplicationId id) NN_NOEXCEPT;

}}  // namespace nn::ns
