﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_StorageId.h>

namespace nn { namespace ns {

/**
* @brief    指定したアプリケーションの実体を削除します。
*
* @details  アプリケーション記録が存在するアプリケーションの実体を削除できます。
*           実体を削除してもアプリケーション記録は削除されずに列挙されます。
*           パッチ・追加コンテンツ・ダウンロード中のデータも一括で削除されます。
*           全ての書き込み可能ストレージから削除されます。
*
* @return   処理の結果が返ります。メディアアクセスエラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
* @retval   ResultApplicationRecordNotFound   指定されたアプリケーション記録が見つかりません。
* @retval   ResultApplicationIsRunning        指定されたアプリケーションが起動中です。
*/
Result DeleteApplicationEntity(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    指定したアプリケーションを完全に削除します。
*
* @details  アプリケーション記録が存在するアプリケーションを完全に削除できます。
*           完全に削除すると、アプリケーション記録も削除されます。
*           パッチ・追加コンテンツ・ダウンロード中のデータも一括で削除されます。
*           全ての書き込み可能ストレージから削除されます。
*           削除時に認識されていなかった書き込み可能ストレージのアプリケーション実体は
*           次回認識時に不要として扱われ、DeleteRedundantApplicationEntity で削除されます。
*
* @return   処理の結果が返ります。メディアアクセスエラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
* @retval   ResultApplicationIsRunning          指定されたアプリケーションが起動中です。
* @retval   ResultGameCardIsInserted            ゲームカードが挿入中のため削除できません。
*/
Result DeleteApplicationCompletely(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    システムが認識している書き込み可能ストレージ内に、不要なアプリケーション実体があるか調べます。
*
* @return   DeleteRedundantApplicationEntity() で削除可能な実体があれば true を返します。
*/
bool IsAnyApplicationEntityRedundant() NN_NOEXCEPT;

/**
* @brief    不要なアプリケーション実体を削除します。
*
* @details  不要なアプリケーション実体とは下記のものを指します。
*           @li アプリケーション記録に存在しない実体
*           @li システムが認識している書き込み可能ストレージ内で、重複している実体
*           バージョンが異なる重複では、バージョンが低いものが削除されます。
*           同じバージョンが重複していた場合は、内蔵ストレージのものが削除されます。
*
* @return   処理の結果が返ります。メディアアクセスエラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
*/
Result DeleteRedundantApplicationEntity() NN_NOEXCEPT;

/**
* @brief    指定したアプリケーション実体が指定のストレージに移動可能か判定します。
*
* @details  アプリケーション記録が存在しない場合は false を返します。
*           ROM カードのデータは考慮されません。
*/
bool IsApplicationEntityMovable(ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT;

/**
* @brief    指定したアプリケーション実体を指定のストレージに移動します。
*
* @details  アプリケーション記録が存在するアプリケーション実体を移動できます。
*           アプリケーションに関連する全てのデータを指定のストレージに移動します。
*           ダウンロード中のデータは一度キャンセルされ、指定のストレージで再開されます。
*           ROM カードのデータは移動されません。
*           実体のみが移動し、アプリケーション記録・アプリケーションビューに影響はありません。
*
* @return   処理の結果が返ります。メディアアクセスエラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
* @retval   ResultStorageSpaceNotEnough         移動先ストレージの空き容量が足りません。
*/
Result MoveApplicationEntity(ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT;

/**
* @brief    指定したアプリケーションの実態が存在するか判定します。
*
* @details  指定したアプリケーションの実態が一部でもシステムが認識しているストレージ内に
*           存在していれば True を返します。
*           アプリケーション記録に存在していない、もしくはシステムが認識しているストレージ内に
*           実態が存在していない場合は false を返します。
*/
bool IsAnyApplicationEntityInstalled(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    指定したアプリケーションが持つ不要な追加コンテンツを削除します。
*
* @details  指定したアプリケーションの追加コンテンツの内、権利のない追加コンテンツを削除します。
*           削除されるコンテンツを他のアプリケーションが参照していた場合、参照していたアプリケーション全てから参照を削除します。
*/
Result CleanupAddOnContentsWithNoRights(ncm::ApplicationId id) NN_NOEXCEPT;

}}  // namespace nn::ns
