﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_Async.h>

namespace nn { namespace ns {

    /**
    * @brief    アプリケーションの更新情報です。
    */
    enum class ApplicationUpdateInfo : Bit8
    {
        UpToDate,   //!< アプリケーションは最新です。
        Updatable   //!< アプリケーションが更新可能です。
    };

    typedef AsyncValue<ApplicationUpdateInfo> AsyncApplicationUpdateInfo;

    /**
    * @brief    アプリケーションの更新情報を非同期処理で取得します。
    *
    * @details  非同期処理をリクエストして AsyncApplicationUpdateInfo を返します。
    *           処理の結果は AsyncApplicationUpdateInfo から取得してください。
    *
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           また、下記の関数と最大非同期処理リクエスト可能数を共有しています。
    *           - RequestUpdateApplication
    *           - RequestUpdateApplicationPatchOnly
    *           - RequestDownloadApplication
    *           - RequestDownloadAddOnContent
    *           - RequestDownloadApplicationControlData
    *           - RequestVerifyApplication
    *           - RequestVerifyAddOnContentsRights
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultApplicationRecordNotFound                 指定されたアプリケーション記録が見つかりません。
    * @retval   ResultOutOfMaxRunningTask                       同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultSystemUpdateRequiredForContentDelivery    （AsyncResult から返される Result）コンテンツ配信に本体更新が必要です。
    * @retval   ResultCanceled                                  （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestApplicationUpdateInfo(AsyncApplicationUpdateInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーション全体の更新を非同期処理で要求します。
    *
    * @details  非同期処理をリクエストして AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功すると、バックグラウンドでアプリケーションの更新が始まります。
    *           存在しない実体も含めて記録されている全ての実体がダウンロードされます。
    *
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           また、下記の関数と最大非同期処理リクエスト可能数を共有しています。
    *           - RequestUpdateApplicationPatchOnly
    *           - RequestApplicationUpdateInfo
    *           - RequestDownloadApplication
    *           - RequestDownloadAddOnContent
    *           - RequestDownloadApplicationControlData
    *           - RequestVerifyApplication
    *           - RequestVerifyAddOnContentsRights
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultApplicationRecordNotFound                 指定されたアプリケーション記録が見つかりません。
    * @retval   ResultAlreadyDownloading                        指定されたアプリケーションが既にダウンロード中、またはパッチ間差分適用中です。
    * @retval   ResultOutOfMaxRunningTask                       同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultAlreadyUpToDate                           （AsyncResult から返される Result）既に対象のコンテンツが最新です。
    * @retval   ResultSystemUpdateRequiredForContentDelivery    （AsyncResult から返される Result）コンテンツ配信に本体更新が必要です。
    * @retval   ResultCanceled                                  （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestUpdateApplication(AsyncResult* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーションのダウンロードをキャンセルします。
    *
    * @details  指定されたアプリケーションがダウンロード中でなくても成功します。
    *
    *           この関数の互換性は維持される必要があります。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
    */
    Result CancelApplicationDownload(ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーションのダウンロードを再開します。
    *
    * @details  下記状態のアプリケーションダウンロードを再開できます。
    *           - ApplicationEntityState == Downloading
    *           - ApplicationDownloadState == Suspended || ApplicationDownloadState == NotEnoughSpace
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound         指定されたアプリケーション記録が見つかりません。
    * @retval   ResultApplicationNotDownloading         指定されたアプリケーションがダウンロード中ではありません。
    * @retval   ResultApplicationDownloadNotResumable   指定されたアプリケーションがダウンロード再開可能ではありません。
    */
    Result ResumeApplicationDownload(ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーションをダウンロードするために必要な空き容量を計算します。
    *           また、容量を開けるべきストレージもあわせて取得します。
    *
    * @details  ApplicationDownloadState が NotEnoughSpace のアプリケーションのみ計算可能です。
    *           outStorageId には容量をあけるべきストレージが入ります。
    *           どのストレージでもよい場合は、StorageId::Any が返ります。
    *           パッチの更新時など、特定のストレージである必要がある場合は SdCard や BuildInUser が返ります。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
    * @retval   ResultApplicationNotDownloading     指定されたアプリケーションがダウンロード中ではありません。
    * @retval   ResultApplicationSeemsEnoughSpace   指定されたアプリケーションが容量不足ではありません。
    */
    Result CalculateApplicationDownloadRequiredSize(ncm::StorageId* outStorageId, int64_t* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーションのパッチ間差分適用をキャンセルします。
    *
    * @details  指定されたアプリケーションがパッチ間差分適用中でなくても成功します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
    */
    Result CancelApplicationApplyDelta(ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーションのパッチ間差分適用を再開します。
    *
    * @details  下記状態の適用タスクを再開できます。
    *           - ApplicationApplyDeltaState == NotEnoughSpace
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound         指定されたアプリケーション記録が見つかりません。
    * @retval   ResultApplicationNotApplying            指定されたアプリケーションがパッチ間差分適用中ではありません。
    * @retval   ResultApplyDeltaTaskNotResumable        指定されたアプリケーションが再開可能ではありません。
    */
    Result ResumeApplicationApplyDelta(ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    パッチ間差分を適用するために必要な空き容量を計算します。
    *           また、容量を開けるべきストレージもあわせて取得します。
    *
    * @details  ApplicationApplyDeltaState が NotEnoughSpace のアプリケーションのみ計算可能です。
    *           outStorageId には容量をあけるべきストレージが入ります。
    *           どのストレージでもよい場合は、StorageId::Any が返ります。
    *           パッチの更新時など、特定のストレージである必要がある場合は SdCard や BuildInUser が返ります。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
    * @retval   ResultApplicationNotApplying        指定されたアプリケーションがパッチ間差分適用中ではありません。
    * @retval   ResultApplicationSeemsEnoughSpace   指定されたアプリケーションが容量不足ではありません。
    */
    Result CalculateApplicationApplyDeltaRequiredSize(ncm::StorageId* outStorageId, int64_t* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /**
     * @brief   Suspended/NotEnoughSpace 状態のタスクをすべて、Resume します。
     *
     * @details 対象となるタスクが無い状態でこの API を呼んでも成功します。
     */
    void ResumeAll() NN_NOEXCEPT;

    /**
     * @brief   システムによる、タスクのコミットを有効化します。
     *
     * @details ダウンロードや適用タスクが完了した際、
     *          コミットが可能な状態であればシステムが自動的にコミットするようになります。
     *          当該アプリが起動中の場合は、コミットが可能ではないと解釈し、コミットは行いません。
     *          自動コミットされなかったタスクは、従来通り View 取得のタイミングでコミットされます。
     *          Enable しても、その時点でコミット可能なものがコミットされたりはしません。
     *
     *          起動時は、システムによるタスクのコミットは無効化されています。
     *          明示的にこの API を呼び出すまで、自動コミットは有効になりません。
     */
     void EnableAutoCommit() NN_NOEXCEPT;

    /**
     * @brief   システムによる、タスクのコミットを無効化します。
     *
     * @details この API の呼び出し以後は、自動コミットが行われません。
     *          自動コミットされない状態の View を取得したい場合は、
     *          まず本 API で自動コミットを無効化し、そのあとで View の取得を行うようにしてください。
     */
     void DisableAutoCommit() NN_NOEXCEPT;

}}  // namespace nn::ns
