﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/npns/npns_Result.h>
#include <nn/nn_Common.h>
#include <nn/account.h>
#include <nn/nn_ApplicationId.h>

namespace nn
{
namespace npns
{
    static const size_t TokenLength = 40;
    static const size_t TopicNameLength = 40;
    static const size_t PayloadMaxLength = 2048;
    static const size_t ReceiverIdLength = 16;
    static const size_t JidLength = 56;
    static const uint16_t Version = 1;

    enum State
    {
        State_Initial,
        State_Idle,
        State_AwaitingOnline,
        State_Online,
        State_NoValidJid,
        State_RequestingJid,
        State_ReadyToConnect,
        State_Connecting,
        State_Connected,
        State_BackoffWaiting,
        State_Suspend,
        State_ShutingDown,
        State_Exit,
        State_AwaitingEulaAccepted,
        State_CheckingArrival,
        State_ConnectedOnHalfAwake
    };

    enum NotificationType
    {
        NotificationType_Normal,
        NotificationType_Topic
    };

    struct ReceiverId
    {
        Bit64 _data[2];

        static ReceiverId Create(const nn::account::Uid& uid)
        {
            ReceiverId receiverId = { { uid._data[0], uid._data[1] } };
            return receiverId;
        }

        // NT の操作主体が無い場合
        static ReceiverId CreateDefault()
        {
            ReceiverId receiverId = { {0, 0} };
            return receiverId;
        }

    };


    struct NotificationToken
    {
        char data[TokenLength];
    };

    // private
    class NotificationData
    {
    public:
        struct Buffer
        {
            uint16_t m_Version;
            uint16_t m_Type;
            uint16_t m_sizeTotal;
            uint16_t m_sizePayload;
            uint64_t m_ApplicationId;
            ReceiverId m_ReceiverId;
            uint64_t m_Flags;
            union
            {
                char m_TopicId[TopicNameLength];
                NotificationToken m_Token;
            };
            char m_Payload[PayloadMaxLength];
        };

        NotificationData()
        {
            m_Data.m_Version = Version;
            m_Data.m_Type = 0;
            m_Data.m_sizeTotal = sizeof(Buffer);
            m_Data.m_sizePayload = 0;
            m_Data.m_ApplicationId = 0;
            m_Data.m_Flags = 0;
        }

        ApplicationId GetApplicationId() const
        {
            ApplicationId appId = {m_Data.m_ApplicationId};
            return appId;
        }

        const ReceiverId& GetReceiverId() const
        {
            return m_Data.m_ReceiverId;
        }

        account::Uid GetUid() const
        {
            account::Uid uid = { {m_Data.m_ReceiverId._data[0], m_Data.m_ReceiverId._data[1]} };
            return uid;
        }

        const char* GetPayload() const
        {
            return m_Data.m_Payload;
        }

        size_t GetPayloadSize() const
        {
            return m_Data.m_sizePayload;
        }

        Buffer m_Data;
    };

}
}

