﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/npns/npns_Types.h>
#include <nn/os.h>
#include <nn/account.h>

namespace nn
{
namespace npns
{
    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知ライブラリの初期化・終了 API
    //! @{

    /**
     * @brief   ライブラリの初期化をおこない、機能を使用可能な状態にします。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *  ライブラリが提供する API は、個別に明示されていないかぎり、利用前にこの関数でライブラリを初期化しておく必要があります。@n
     *
     *  ライブラリがすでに初期化された状態でも重ねて呼ぶことができますが、その場合、ライブラリの利用を完全に終了するにはこの関数を呼んだのと同じ回数だけ
     *  @ref nn::npns::Finalize 関数を呼ぶ必要があります。
     */
    nn::Result Initialize() NN_NOEXCEPT;

    /**
     * @brief   ライブラリが初期化済みかを判定します。
     *
     * @return  ライブラリがすでに初期化されていれば true を、まだ初期化されていなければ false を返します。
     *
     * @details
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief   ライブラリの利用を終了します。
     *
     * @return  処理の結果が返ります。
     *
     * @details
     *  @ref nn::npns::Initialize 関数を複数回呼んでいる場合、ライブラリの利用を完全に終了するには同じ回数だけこの関数を呼ぶ必要があります。@n
     *
     *  ライブラリの利用が完全に終了されると、システムで確保していたリソースが解放されます。@n
     *
     *  ライブラリが未初期化の状態でコールしても副作用はありません。
     */
    void Finalize() NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知における受信 API
    //! @{

    /**
     * @brief   （未実装）自プロセスが受信可能な全ての通知を配送するよう指示します。
     *
     * @return  処理の結果が返ります。
     *
     * @details 通知を受信するにはどの宛先の通知を受信するかを明示する必要があります。この API では meta で指定された宛先すべての通知を配送するように設定します。
     *
     */
    nn::Result ListenAll() NN_NOEXCEPT;

    /**
     * @brief   自プロセスが受信可能な通知のうち、特定の宛先を持つ通知のみを配送するように明示的に指示します。
     *
     * @param[in] ApplicationId     受信対象のアプリケーション ID
     *
     * @return  処理の結果が返ります。
     *
     * @details 通知を受信するにはどの宛先の通知を受信するかを明示する必要があります。この API では meta で指定された宛先のうちどれを自プロセスに配送するのかを個別に設定します。
     *
     *          @ref ListenToMyApplicationId() が利用可能な場面では、セキュリティの問題上そちらを利用してください。
     */
    nn::Result ListenTo(ApplicationId toId) NN_NOEXCEPT;

    /**
     * @brief   自身のアプリケーション向けの通知を配送するように明示的に指示します。
     *
     * @return  処理の結果が返ります。
     *
     * @details 通知を受信するにはどの宛先の通知を受信するかを明示する必要があります。
     *          この API では meta に記載されたアプリケーション ID 向けの通知を受信できるようになります。
     */
    nn::Result ListenToMyApplicationId() NN_NOEXCEPT;

    /**
     * @brief   自プロセス向けに配送される通知を受信します。
     *
     * @param[out] pOut             受信した通知の格納先
     *
     * @return  処理の結果が返ります。
     *
     * @details 自プロセス向けに配送された通知はいったんシステム内部のキューに格納されます。この API はキューから通知データを一つずつ取り出します。
     *
     */
    nn::Result Receive(NotificationData* pOut) NN_NOEXCEPT;

    /**
     * @brief   自プロセス向けの通知が配送されたことを通知するイベント @ref nn::os::SystemEvent を取得します。
     *
     * @param[out] event            イベントの格納先
     *
     * @return  処理の結果が返ります。
     *
     * @details 取得したイベントは、自プロセス向けの通知がシステム内部のキューに格納されたタイミングでシグナルされます。@n
     * イベントの種別は自動リセットです。シグナルを検知したら、通知の取りこぼしを防ぐために @ref Receive 関数でキューが空になったことを確認するようにしてください。
     *
     */
    nn::Result GetReceiveEvent(nn::os::SystemEvent& event) NN_NOEXCEPT;
    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知におけるトークン管理 API
    //! @{

    /**
     * @brief   通知トークンを作成します
     *
     * @param[out] pTokenOut            作成された通知トークンの格納先
     * @param[in] uid                   通知の宛先となるユーザ識別子（ユーザを区別しない場合は 0 を指定します）
     *
     * @return  処理の結果が返ります。
     * @retval  ResultSuccess                   成功
     * @retval  ResultAccountNotAvailable       ID が未作成
     *
     * @details 自身のアプリケーションかつ指定の uid 向けの通知トークンの作成をサーバに要求します。オフラインでは使用できません。
     *
     */
    nn::Result CreateToken(NotificationToken* pTokenOut, const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief   通知トークンを削除します
     * @param[in] uid                   削除対象となるユーザ識別子
     *
     * @return  処理の結果が返ります。
     *
     * @details 自身のアプリケーションかつ指定の uid 向けに作られた通知トークンの削除をサーバに要求します。オフラインでは使用できません。
     *
     */
    nn::Result DestroyToken(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief   （未実装）通知トークンの有効性を確認します
     *
     * @return  処理の結果が返ります。
     *
     * @details 通知トークンが有効かどうかをサーバに確認します。トークンが無効な場合はトークン経由での通知は発行できなくなります。オフラインでは使用できません。
     *
     */
    nn::Result QueryIsTokenValid(bool& isValid, const NotificationToken& token) NN_NOEXCEPT;

    //! @}

    ///////////////////////////////////////////////////////////////////////////
    //! @name プッシュ通知における状態取得 API
    //! @{

    /**
     * @brief   通知デーモンの状態 @ref nn::npns::State を取得します。
     *
     * @return  通知デーモンの状態が返ります。
     *
     * @details 事前条件にデーモンの状態が含まれる場合は、この API を使って状態の変化を待ってください。@n
     *
     */
    nn::npns::State GetState() NN_NOEXCEPT;

    /**
     * @brief   通知デーモンの状態 @ref nn::npns::State が変わったタイミングでシグナルされるイベント @ref nn::os::SystemEvent を取得します。
     *
     * @param[out] event            イベントの格納先
     *
     * @return  処理の結果が返ります。
     *
     * @details 取得したイベントは、デーモンの状態 @ref nn::npns::State が変わるたびにシグナルされます。
     * イベントの種別は自動リセットです。
     *
     */
    nn::Result GetStateChangeEvent(nn::os::SystemEvent& event) NN_NOEXCEPT;

    //! @}
}
}
