﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/os/os_Mutex.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace nim { namespace srv {

    class SimpleDataStoreImpl
    {
    public:
        SimpleDataStoreImpl() NN_NOEXCEPT : m_Mutex(false) {}
        ~SimpleDataStoreImpl() NN_NOEXCEPT;

        Result Initialize(
            const char* mountName,
            fs::SystemSaveDataId id, int64_t saveDataSize, int64_t saveDataJournalSize, int saveDataFlags,
            const void* initialData, size_t dataSize) NN_NOEXCEPT;

        Result Load(void* data, size_t size) NN_NOEXCEPT;
        Result Save(const void* data, size_t size) NN_NOEXCEPT;
        Result Clear() NN_NOEXCEPT;

    private:
        os::Mutex m_Mutex;
        typedef kvdb::BoundedString<64> Path;
        util::optional<Path> m_MountName;

        void* m_Data;
        size_t m_Size;

        Path MakeDataFilePath() NN_NOEXCEPT;
    };

    template<typename T>
    class SimpleDataStore : private SimpleDataStoreImpl
    {
        NN_STATIC_ASSERT(std::is_trivially_copyable<T>::value);
    public:
        SimpleDataStore() NN_NOEXCEPT : m_Data() {}
        Result Initialize(const char* mountName, fs::SystemSaveDataId id, int64_t saveDataSize, int64_t saveDataJournalSize, int saveDataFlags) NN_NOEXCEPT
        {
            SetInitialData(&m_Data);
            NN_RESULT_DO(SimpleDataStoreImpl::Initialize(
                mountName,
                id, saveDataSize, saveDataJournalSize, saveDataFlags,
                &m_Data, sizeof(m_Data)));
            NN_RESULT_DO(Load(&m_Data, sizeof(m_Data)));
            NN_RESULT_SUCCESS;
        }

    protected:
        T Get() const NN_NOEXCEPT
        {
            return m_Data;
        }

        T& GetRef() NN_NOEXCEPT
        {
            return m_Data;
        }

        Result Save() NN_NOEXCEPT
        {
            return SimpleDataStoreImpl::Save(&m_Data, sizeof(m_Data));
        }

    private:
        virtual void SetInitialData(T*) NN_NOEXCEPT {}

        T m_Data;
    };

}}}
