﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/util/util_Optional.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/nim/srv/nim_DeviceContext.h>

namespace nn { namespace nim { namespace srv {

    struct HttpHeaderValue
    {
        char string[128];
    };

    Result FindHttpHeader(util::optional<HttpHeaderValue>* outValue, const char* name, const char* buffer, size_t bufferSize) NN_NOEXCEPT;

    class HttpConnection
    {
    public:
        HttpConnection() NN_NOEXCEPT : m_TimeoutSecond(60) {}
        ~HttpConnection() NN_NOEXCEPT;

        typedef std::function<Result(const char* buffer, size_t bufferSize)> HeaderCallback;
        typedef std::function<Result(const void* buffer, size_t bufferSize)> WriteCallback;

        Result Initialize(DeviceContext* deviceContext) NN_NOEXCEPT;
        Result Initialize(ssl::Context* sslContext) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        Result Head(const char* url, HeaderCallback headerCallback, const char* headerList[] = nullptr, int headerCount = 0, int timeoutSecond = DefaultTimeoutSecond) NN_NOEXCEPT;
        Result Get(const char* url, WriteCallback writeCallback, const char* headerList[] = nullptr, int headerCount = 0, int timeoutSecond = DefaultTimeoutSecond) NN_NOEXCEPT;
        Result Get(const char* url,  HeaderCallback headerCallback, WriteCallback writeCallback, const char* headerList[] = nullptr, int headerCount = 0, int timeoutSecond = DefaultTimeoutSecond) NN_NOEXCEPT;
        Result Post(const char* url, const char* postFields, WriteCallback writeCallback, const char* headerList[] = nullptr, int headerCount = 0, int timeoutSecond = DefaultTimeoutSecond) NN_NOEXCEPT;
        Result Post(const char* url, const char* postFields, HeaderCallback headerCallback, WriteCallback writeCallback, const char* headerList[] = nullptr, int headerCount = 0, int timeoutSecond = DefaultTimeoutSecond) NN_NOEXCEPT;
        static Result Escape(char* buffer, size_t bufferSize, const char* str) NN_NOEXCEPT;

        void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

        void Cancel() NN_NOEXCEPT
        {
            m_IsCanceled = true;
        }
        void ResetCancel() NN_NOEXCEPT
        {
            m_IsCanceled = false;
        }
        bool IsCanceled() const NN_NOEXCEPT
        {
            return m_IsCanceled;
        }
        void SetLastResult(Result result) NN_NOEXCEPT
        {
            m_LastResult = result;
        }
        int GetLastStatusCode() const NN_NOEXCEPT
        {
            return m_LastStatusCode;
        }

        void OnProgress(int64_t dltotal, int64_t  dlnow, int64_t  ultotal, int64_t ulnow) NN_NOEXCEPT;
        bool IsTimeout() NN_NOEXCEPT
        {
            return m_IsTimeout;
        }

        const char* GetUrl() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT
        {
            return (nullptr != m_Impl);
        }

        //!----------------------------------------------------------------------------
        //! @name パラメータ分散設定方式 Perform
        //! @{
        struct TransactionHandle
        {
            void*   pHeaders;
            char    pCustomMethod[16];
            bool    hasPostField;
            bool    isHttpsUrl;

            void Reset() NN_NOEXCEPT;

            NN_IMPLICIT TransactionHandle() NN_NOEXCEPT : pHeaders(nullptr)
            {
                Reset();
            }

            ~TransactionHandle() NN_NOEXCEPT
            {
                Reset();
            }

            bool HasCustomMethod() const NN_NOEXCEPT
            {
                return '\0' != pCustomMethod[0];
            }
        };

        Result BeginTransaction(TransactionHandle* pHandle, bool withResetCancel = true) NN_NOEXCEPT;

        Result SetTransactionUrl(TransactionHandle* pHandle, const char* pUrl) NN_NOEXCEPT;
        Result SetTransactionHeader(TransactionHandle* pHandle, const char* pHeader) NN_NOEXCEPT;
        Result SetTransactionPostFields(TransactionHandle* pHandle, const char* pFields, const int64_t size, bool isClone) NN_NOEXCEPT;
        NN_FORCEINLINE Result SetTransactionPostFields(TransactionHandle* pHandle, const char* pFields) NN_NOEXCEPT
        {
            return SetTransactionPostFields(pHandle, pFields, -1, false);
        }
        //! @details    カスタムメソッドに指定された文字列を判断した特殊処理は行いません。@n
        //!             例えば、PUT 利用時、リクエストボディがない場合の要求でも、空ボディを @ref SetTransactionPostFields() に設定してください。@n@n
        //!             CURL における PUT (upload) 機能の制約です。
        //!             e.g.)@n
        //!                 NN_FUNCTION_LOCAL_STATIC(char, s_Empty, [] = "");@n
        //!                 SetTransactionPostFields(s_Empty);@n
        Result SetTransactionCustomMethod(TransactionHandle* pHandle, const char* pMethod) NN_NOEXCEPT;

        Result EndTransaction(TransactionHandle* pHandle, util::optional<HeaderCallback> headerCallback, util::optional<WriteCallback> writeCallback, int timeoutSecond) NN_NOEXCEPT;
        void CleanTransaction(TransactionHandle* pHandle) NN_NOEXCEPT;
        //! @}
        //!----------------------------------------------------------------------------

    private:
        static const int DefaultTimeoutSecond = 60;

        Result InitializeCore(ssl::Context* sslContext) NN_NOEXCEPT;
        Result Perform(const char* url, const char* postFields, util::optional<HeaderCallback> headerCallback, util::optional<WriteCallback> writeCallback, const char * headerList[], int headerCount, int timeoutSecond) NN_NOEXCEPT;
        void ResetProgress(int timeoutSecond) NN_NOEXCEPT;
        void* m_Impl{};
        std::atomic_bool m_IsCanceled{};
        bool m_IsTimeout{};
        Result m_LastResult;
        int m_LastStatusCode;
        DeviceContext* m_DeviceContext{};

        int64_t m_LastDownloaded{};
        int64_t m_LastUploaded{};
        TimeSpan m_LastProgressChangedTime{};
        int m_TimeoutSecond;
    };
}}}
