﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <atomic>
#include <cstring>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace nim { namespace srv {

namespace detail {

/**
 * @brief   非特殊化対象構造体用 std::atomic 代替クラス。
 * @details mutex ロックによる高コスト atomic です。@n
 *          メモリオーダーなどはインタフェース互換のための宣言です。@n
 *          std::atomic<T> で非特殊化対象外構造体を用いた場合、各プラットフォームで以下のような制限があります。
 *          Clang:
 *              @li 8byte以上の非特殊化対象外構造体を利用した場合、実行時にアライン違反メモリアクセスが発生するケースがあります。( 恐らくGCC依存 )
 *                  alignas() なども効果がなかった。
 *          VC2015:
 *              @li 構造体長が 2/4/8 の時は、alignof(T) >= sizeof(T) である必要があります。
 *              @li x86(32bit)では、8byte アラインの構造体が C2719 エラーで使えない。( 普通に不具合と思われる。 )
 */
template<typename TValue>
class MutexAtomic
{
public:
    explicit MutexAtomic() NN_NOEXCEPT : m_Value(), m_Lock() {}
    explicit MutexAtomic(const TValue& source) NN_NOEXCEPT : m_Value(source), m_Lock() {}

    TValue load(const std::memory_order) const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        return m_Value;
    }

    TValue load(const std::memory_order) const volatile NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        return m_Value;
    }

    void store(const TValue& source, const std::memory_order) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        m_Value = source;
    }

    void store(const TValue& source, const std::memory_order) volatile NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        m_Value = source;
    }

    TValue exchange(const TValue& desired, const std::memory_order order = std::memory_order_seq_cst) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        auto preValue = m_Value;
        m_Value = desired;
        return preValue;
    }

    TValue exchange(const TValue& desired, const std::memory_order order = std::memory_order_seq_cst) volatile NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        auto preValue = m_Value;
        m_Value = desired;
        return preValue;
    }

    bool compare_exchange_strong(TValue& expected, TValue desired, const std::memory_order order = std::memory_order_seq_cst) volatile NN_NOEXCEPT
    {
        NN_UNUSED(order);
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        const bool r = (0 == std::memcmp(&m_Value, &expected, sizeof(TValue)));
        if (r)
        {
            m_Value = desired;
        }
        else
        {
            expected = m_Value;
        }
        return r;
    }

    bool compare_exchange_strong(TValue& expected, TValue desired, const std::memory_order order = std::memory_order_seq_cst) NN_NOEXCEPT
    {
        NN_UNUSED(order);
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        const bool r = (0 == std::memcmp(&m_Value, &expected, sizeof(TValue)));
        if (r)
        {
            m_Value = desired;
        }
        else
        {
            expected = m_Value;
        }
        return r;
    }

    bool compare_exchange_strong(TValue& expected, TValue desired, const std::memory_order, const std::memory_order) volatile NN_NOEXCEPT
    {
        return compare_exchange_strong(expected, desired);
    }

    bool compare_exchange_strong(TValue& expected, TValue desired, const std::memory_order, const std::memory_order) NN_NOEXCEPT
    {
        return compare_exchange_strong(expected, desired);
    }

private:
    TValue                          m_Value;
    mutable os::SdkRecursiveMutex   m_Lock;
};

} // namespace detail

}}} // namespace nn::nim::srv
