﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_TimeSpan.h>
#include <nn/account/account_Types.h>
#include <nn/ec/ec_ShopServiceAccessorTypes.h>

namespace nn { namespace nim {

    struct ShopServiceAccessTypes
    {
        /**
         * @brief   要求サーバー完全修飾ドメイン(FQDN)種別
         */
        typedef ::nn::Bit8  Server;

        /**
         * @brief   サーバー要求HTTPメソッド
         */
        typedef ::nn::Bit8  Method;

        /**
         * @brief   要求パラメータ転送用固定長パッケージ
         */
        struct NN_ALIGNAS(8) FixedParams
        {
            ::nn::account::Uid      uid;            //!< ユーザアカウントID
            ::nn::TimeSpanType      timeout;        //!< 通信タイムアウト
            Method                  method;         //!< メソッド
            Bit8                    headerCount;    //!< 追加ヘッダ総数
            Bit16                   reserved16;
            Bit32                   reserved32;
        };

        /**
         * @brief   構造体生成ユーティリティ。
         */
        static inline Server CreateServerFrom(const ::nn::ec::ShopService::Type& server_) NN_NOEXCEPT
        {
            return static_cast<Server>(server_);
        }
    };

    NN_STATIC_ASSERT(sizeof(::nn::nim::ShopServiceAccessTypes::Server) == 1);
    NN_STATIC_ASSERT(std::is_pod<::nn::nim::ShopServiceAccessTypes::Server>::value);

    NN_STATIC_ASSERT(sizeof(::nn::nim::ShopServiceAccessTypes::Method) == 1);
    NN_STATIC_ASSERT(std::is_pod<::nn::nim::ShopServiceAccessTypes::Method>::value);

    NN_STATIC_ASSERT(sizeof(::nn::nim::ShopServiceAccessTypes::FixedParams) == 32);
    NN_STATIC_ASSERT(std::is_pod<::nn::nim::ShopServiceAccessTypes::FixedParams>::value);

    /**
     * @brief   複数 null 終端文字列のシリアライズ/デシリアライズユーティリティ。
     *
     * @note    バイナリPOSTデータを採用したい場合は以下のような拡張を行う想定。
     *          この手法なら後発拡張でも上位互換可能。 ( ※カテゴリ定義が連番列挙の間なら )
     *          尚、バイナリPOSTデータの場合は、curl_easy_setopt() で CURLOPT_POSTFIELDSIZE にデータサイズが指定されている事を確認する事。
     *
     *          @li CategoryType の値を列挙値ではなく、8bit ビットフィールド構成とする。
     *              上位 3bit を制御用。
     *                  0: Ascii
     *                  1: Binary
     *                      :
     *                  7: ???
     *              下位 5bit を列挙値。
     *
     *          @li シリアライザ / デシリアライザではカテゴリチェックの際に上位3bitをチェックして、Ascii(0) なら従来通りとする。
     *              Binary(1) なら、以下フォーマット。
     *                  | カテゴリ | サイズ | データ | Terminate |
     *                  | 1 byte   | 2 byte | n byte | 1 byte    |
     *
     *          @li CategoryType::BinaryPostData = 0x20 を追加する。
     */
    struct SerializedStrings
    {
        /**
         * @brief   @ref Append() のエラー定義。
         */
        enum class ErrorCause : Bit8
        {
            None,
            CapacityOver,
            InvalidCharacter,
        };

        /**
         * @brief   文字列要素種別判断カテゴリ定義。
         */
        struct CategoryType
        {
            static const char Terminate = 0;
            static const char Path = 1;
            static const char Header = 2;
            static const char PostData = 3;
        };

        /**
         * @brief   複数 null 終端文字列のシリアライズ/デシリアライズユーティリティ。
         *
         * @details シリアライズとデシリアライズを同時には行えません。@n
         *          一方の目的でバッファを利用してください。@n
         *          各追加文字列毎のnull終端もシリアライズ対象になります。@n
         *
         *          nim( server ), ec( client ) の双方で IPC 通信メッセージとして利用するため、本ファイル中でのソースコード実装です。@n
         *          ※ cpp 分離しても良いけども、client( ec ) が nim のソースファイルを .a に含むように !.nact 記載するのは憚られた。
         */
        template<typename TCharacter, typename TCapacity>
        class Builder
        {
        public:
            static const TCharacter Termination = 0;

            /**
             * @brief       コンストラクタ
             */
            explicit Builder(TCharacter* pBuffer, TCapacity capacity, bool withClean = true) NN_NOEXCEPT
                : m_pTop(pBuffer), m_pIndex(pBuffer), m_Capacity(capacity)
            {
                NN_SDK_REQUIRES_NOT_NULL(pBuffer);
                NN_SDK_REQUIRES(capacity > 0);
                if (withClean)
                {
                    *pBuffer = Termination;
                }
            }

            /**
             * @brief       カレントインデクスマーカーを先頭に戻します。
             */
            void Rewind() NN_NOEXCEPT
            {
                m_pIndex = m_pTop;
            }

            /**
             * @brief       先頭からカレントインデクスマーカーが示す位置までの文字列数を返します。
             * @return      文字列数を返します。全データ終端を示す Termination も1つとして含まれます。
             */
            const TCapacity GetCount() const NN_NOEXCEPT
            {
                return (m_pIndex - m_pTop) + 1;
            }

            /**
             * @brief       バッファ先頭を返します。
             */
            const TCharacter* GetTop() const NN_NOEXCEPT
            {
                return m_pTop;
            }

            /**
             * @brief       末尾へ文字列を追加します。
             *
             * @param[in]   category    追加文字列の種別を表すカテゴリヘッダ文字を指定します。 0 指定は禁止です。
             * @param[in]   pSource     追加したい null 終端文字列を指定します。
             * @param[in]   checker     文字チェック評価式を指定します。( bool func(char) 書式 )
             *
             * @return      処理結果が返されます。
             * @retval      ErrorCause::None                処理に成功しました。
             * @retval      ErrorCause::CapacityOver        上限容量を越えました。
             * @retval      ErrorCause::InvalidCharacter    追加文字列中に無効な文字を検出しました。
             *
             * @details     上限容量を越えた場合は入力は捨てられます。@n
             *              バッファ末尾は必ず Termination が付与されます。
             */
            template<typename ValidationChecker>
            ErrorCause Append(const TCharacter category, const TCharacter* pSource, ValidationChecker checker) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES(Termination != category);

                if (nullptr == pSource)
                {
                    return ErrorCause::None;
                }

                // バッファ末尾 [index] は必ず Termination にする。
                auto errorCause = ErrorCause::CapacityOver;
                const auto pEnd = &m_pTop[m_Capacity - 1];
                auto pIndex = m_pIndex;

                if (pIndex < pEnd)
                {
                    *pIndex++ = category;
                    while (pIndex < pEnd)
                    {
                        const auto data = *pSource++;
                        if ('\0' == data)
                        {
                            *pIndex++ = data;
                            errorCause = ErrorCause::None;
                            break;
                        }
                        else if (!checker(data))
                        {
                            errorCause = ErrorCause::InvalidCharacter;
                            break;
                        }
                        *pIndex++ = data;
                    }
                    m_pIndex = pIndex;
                }
                *pIndex = Termination;
                return errorCause;
            }

            /**
             * @brief       カレントインデクスから null 終端文字列を抽出します。
             *
             * @details     抽出後はカレントインデクスが次の文字列を示します。@n
             *              データ終了判断は @ref pOutCategory に Termination が入った時が終わりです。
             */
            void Extract(TCharacter* pOutCategory, TCharacter* pOutValue, const TCapacity receivableSize) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutCategory);
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                NN_SDK_REQUIRES(receivableSize > 0);

                const auto pEnd = &m_pTop[m_Capacity - 1];
                auto pIndex = m_pIndex;

                // 先頭が Termination なら、抽出対象データはない。
                const auto topCode = *pIndex++;
                *pOutCategory = topCode;
                *pOutValue = '\0';

                if (Termination == topCode || pIndex >= pEnd)
                {
                    return;
                }
                const auto pOutEnd = &pOutValue[receivableSize - 1];
                TCharacter value;
                do
                {
                    value = *pIndex++;
                    *pOutValue++ = value;
                    if (pOutValue >= pOutEnd)
                    {
                        // 受信バッファ不足ならインデクスは更新しない。
                        *pOutValue = '\0';
                        return;
                    }
                    if (pIndex < pEnd)
                    {
                        *pOutValue = '\0';
                        break;
                    }
                } while ('\0' != value);
                m_pIndex = pIndex;
            }

            /**
             * @brief       カレントインデクスから null 終端文字列を抽出します。
             *
             * @param[out]  pOutValue   カレントインデクスが示す null 終端文字列の先頭アドレス。
             * @return      抽出文字列の種別を表すカテゴリヘッダ文字。@n
             *              @ref Termination の場合はカレントインデクスがデータ終端示します。
             *
             * @details     抽出後はカレントインデクスが次の文字列を示します。
             */
            TCharacter FindNext(TCharacter** pOutValue) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);

                const auto pEnd = &m_pTop[m_Capacity - 1];
                auto pIndex = m_pIndex;

                // 先頭が Termination なら、抽出対象データはない。
                const auto topCode = *pIndex++;
                *pOutValue = nullptr;

                if (Termination == topCode || pIndex >= pEnd)
                {
                    return Termination;
                }
                *pOutValue = pIndex;

                // 次の先頭へインデクスを更新
                TCharacter value;
                do
                {
                    value = *pIndex++;
                    if (pIndex < pEnd)
                    {
                        break;
                    }
                } while ('\0' != value);
                m_pIndex = pIndex;
                return topCode;
            }

        private:
            TCharacter * const   m_pTop;
            TCharacter*         m_pIndex;
            TCapacity           m_Capacity;
        };
    };

}}
