﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/es/es_ELicenseTypes.h>
#include <nn/es/es_Types.h>

namespace nn { namespace nim {

/**
   @brief 付与されるライセンスの種類です。
 */
enum class ELicenseType : uint8_t
{
    Unknown,                     //!< 不明なライセンス (意図としては、新ファームでは認識できるが、今のファームでは認知できない場合)
    DeviceLinkedPermanent,       //!< 機器認証に基づく permanent ライセンス
    Permanent,                   //!< 動的な permanent ライセンス
    AccountRestrictivePermanent, //!< デバイス共有不可な、動的な permanent ライセンス
    Temporary,                   //!< 動的な temporary ライセンス
    Unavailable,                 //!< 利用可能なライセンスがありません
    EndOfContents                // 末尾判断定数。利用可能な値ではありません。
};


/**
   @brief 問い合わせたライセンスの状態です。
*/
enum class ELicenseStatus : uint8_t
{
    Unknown,                                                //!< 不明なライセンス (意図としては、新ファームでは認識できるが、今のファームでは認知できない場合)
    Assignable,                                             //!< 割り当て可能です。
    AlreadyAssigned,                                        //!< 現在は使用されません。( @deprecated )
    NotAssignableSinceTitleDoesNotSupportDynamicRights,     //!< 現在は使用されません。( @deprecated )
    NotAssignableSinceLimitExceeded,                        //!< サーバ上権利は存在しますが、付与上限に達しているためこのデバイスでは利用できません。
    NotAssignableSinceNoRights,                             //!< 権利を所有していません。
    NotAssignableSinceNotDeviceLinked,                      //!< サーバ上権利は存在しますが、機器認証でのみ取得可能な権利であるためこのデバイスでは利用できません。
    EndOfContents                                           // 末尾判断定数。利用可能な値ではありません。
};

/**
   @brief 利用できる可能性がある eLicense 情報です。
          実際にそのデバイスに割り当て可能かは、LicenseStatus を参照してください。
          この情報はあくまでもサーバの状態であり、デバイス上のライセンス状態とは無関係です。
          (サーバ上で AlreadyAssigned となっていても、クライアント上にその eLicense があるとは限らない)
*/
struct AvailableELicense
{
    es::RightsId                rightsId;       //!< 権利の ID
    account::NintendoAccountId  naId;           //!< 利用対象となるユーザー
    ELicenseType                licenseType;    //!< 利用可能なライセンスの種類
    ELicenseStatus              licenseStatus;  //!< ライセンスの状態
    Bit8                        reserved[6];
};

/**
   @brief AssignELicense で割り当てられた eLicense の情報です。
*/
struct AssignedELicense
{
    es::ELicenseId              eLicenseId;     //!< ライセンスの ID
    es::RightsId                rightsId;       //!< 権利の ID
    account::NintendoAccountId  naId;           //!< 権利所有ユーザー
    ELicenseType                licenseType;    //!< 付与されたライセンスの種類
    Bit8                        reserved[7];
};

/**
   @brief eLicense の無効化理由です。
 */
enum class RevokeReason : uint8_t
{
    Unknown,                 //!< 無効化された理由が不明です (意図としては、新ファームでは認識できるが、今のファームでは認知できない場合)
    Expired,                 //!< 期限が切れました
    OtherDeviceAssigned,     //!< 他のデバイスに権利が割り当てられました
    DuplicateUsageDetected,  //!< 同時利用が検出されました
    EndOfContents            // 末尾判断定数。利用可能な値ではありません。
};

}}
