﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/nim/detail/nim_IAsync.sfdl.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace nim {
    namespace detail {
        template <typename IAsyncT>
        class AsyncBase
        {
        public:
            void Initialize(sf::SharedPointer<IAsyncT> sp, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
            {
                m_Event.emplace();
                m_Event->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), os::EventClearMode_ManualClear);
                m_Async = sp;
            }

            virtual ~AsyncBase() NN_NOEXCEPT
            {
                if (m_Async)
                {
                    Finalize();
                }
            }

            void Wait() NN_NOEXCEPT
            {
                m_Event->Wait();
            }

            bool TryWait() NN_NOEXCEPT
            {
                return m_Event->TryWait();
            }

            void Cancel() NN_NOEXCEPT
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Async->Cancel());
            }

            void Finalize() NN_NOEXCEPT
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Async->Cancel());
                m_Async = nullptr;
                m_Event = util::nullopt;
            }

            os::SystemEvent& GetEvent() NN_NOEXCEPT
            {
                return *m_Event;
            }

            void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_Async->GetErrorContext(outValue));
            }

        protected:
            sf::SharedPointer<IAsyncT> m_Async;

        private:
            util::optional<os::SystemEvent> m_Event;
        };

        template <typename T>
        class AsyncValue : public AsyncBase<detail::IAsyncValue>
        {
        public:
            Result Get(T* outValue) NN_NOEXCEPT
            {
                Wait();
                return m_Async->Get(sf::OutBuffer(reinterpret_cast<char*>(outValue), sizeof(T)));
            }
        };

        /**
         * @brief   インデクスオフセット指定非対応版 AsyncValueList クラス。
         *
         * @details 本インタフェースは offset 非対応の Legacy な ValueList 形式クラスです。
         *          新たに ValueList 形式を利用する場合は、新世代の @ref AsyncValueList を推奨します。
         */
        template <typename T>
        class AsyncValueListLegacy : public AsyncBase<detail::IAsyncValue>
        {
        public:
            Result Get(int* outCount, T outList[], int count) NN_NOEXCEPT
            {
                Wait();
                uint64_t size;
                NN_RESULT_DO(m_Async->GetSize(&size));
                NN_RESULT_DO(m_Async->Get(sf::OutBuffer(reinterpret_cast<char*>(outList), sizeof(T) * count)));
                *outCount = static_cast<int>(size / sizeof(T));
                NN_RESULT_SUCCESS;
            }
        };

        /**
         * @brief   インデクスオフセット指定対応版 AsyncValueList クラス。
         *
         * @details 取得可能な総要素数の取得インタフェース、及び、インデクスオフセット指定の取得インタフェースを提供します。
         */
        template <typename T>
        class AsyncValueList : public AsyncBase<detail::IAsyncData>
        {
        public:
            //! @brief      総要素数の取得。
            //! @param[out] pOutCount       取得可能な総要素数。
            Result GetSize(int* pOutCount) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutCount);

                Wait();
                int64_t outByteSize;
                NN_RESULT_DO(m_Async->GetSize(&outByteSize));
                *pOutCount = static_cast<int>(outByteSize / sizeof(T));
                NN_RESULT_SUCCESS;
            }

            //! @brief      結果要素リスト取得( オフセット指定式 )。
            //!
            //! @param[out] pOutCount       @a pOutList へ出力された実際の要素数( 配列要素単位 )。
            //! @param[out] pOutList        出力先バッファ。
            //! @param[in]  listCapacity    出力先バッファ容量( 配列要素単位 )。
            //! @param[in]  offset          取得対象先頭インデクスオフセット( 配列要素単位 )。
            //!
            //! @return     非同期要求処理の結果を返します。@n
            //!             非同期要求で発生した通信結果及び通信コンテキスト生成処理での成否 Result も含まれます。
            //!
            //! @retval     nn::ResultSuccess               正常に処理を完了しました。
            //! @retval     nn::nim::ResultInvalidArgument  不正な引数指定が行われました。( 事前条件を満たしていれば発生しません )
            //!
            //! @pre
            //!     - nullptr != pOutCount
            //!     - nullptr != pOutList
            //!     - listCapacity > 0
            //!     - offset >= 0
            //!
            //! @details    結果要素配列中の @a offset で指定された要素から @a listCapacity で指定された要素数分を @a pOutList へ取得します。
            Result Get(int* pOutCount, T pOutList[], int listCapacity, int offset) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutCount);
                NN_SDK_REQUIRES_NOT_NULL(pOutList);
                NN_SDK_REQUIRES(listCapacity > 0);
                NN_SDK_REQUIRES(offset >= 0);

                Wait();
                uint64_t outByteSize;
                const auto byteOffset = static_cast<int64_t>(offset * sizeof(T));
                const auto byteCapacity = static_cast<size_t>(sizeof(T) * listCapacity);
                NN_RESULT_DO(m_Async->Read(&outByteSize, byteOffset, sf::OutBuffer(reinterpret_cast<char*>(pOutList), byteCapacity)));
                *pOutCount = static_cast<int>(outByteSize / sizeof(T));
                NN_RESULT_SUCCESS;
            }

            //! @brief      結果要素リスト取得。
            //!
            //! @param[out] pOutCount       @a pOutList へ出力された実際の要素数( 配列要素単位 )。
            //! @param[out] pOutList        出力先バッファ。
            //! @param[in]  listCapacity    出力先バッファ容量( 配列要素単位 )。
            //!
            //! @return     非同期要求処理の結果を返します。@n
            //!             非同期要求で発生した通信結果及び通信コンテキスト生成処理での成否 Result も含まれます。
            //!
            //! @retval     nn::ResultSuccess               正常に処理を完了しました。
            //! @retval     nn::nim::ResultInvalidArgument  不正な引数指定が行われました。( 事前条件を満たしていれば発生しません )
            //!
            //! @pre
            //!     - nullptr != pOutCount
            //!     - nullptr != pOutList
            //!     - listCapacity > 0
            //!     - offset >= 0
            //!
            //! @details    結果要素配列中の先頭から @a listCapacity で指定された要素数分を @a pOutList へ取得します。
            Result Get(int* pOutCount, T pOutList[], int listCapacity) NN_NOEXCEPT
            {
                return Get(pOutCount, pOutList, listCapacity, 0);
            }
        };
    }

    class AsyncResult : public detail::AsyncBase<detail::IAsyncResult>
    {
    public:
        Result Get() NN_NOEXCEPT
        {
            Wait();
            return m_Async->Get();
        }
    };

    class AsyncData : public detail::AsyncBase<detail::IAsyncData>
    {
    public:
        Result Get() NN_NOEXCEPT
        {
            Wait();
            return m_Async->Get();
        }

        int64_t GetSize() NN_NOEXCEPT;

        Result Read(size_t* outValue, int64_t offset, void* buffer, size_t bufferSize) NN_NOEXCEPT;

        Result GetETag(ETag* outValue) NN_NOEXCEPT;
    };

}}
