﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesWirelessSetting.h>
#include <nn/nifm/nifm_TypesIpSetting.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_StringUtil.h>

namespace nn
{
namespace nifm
{

/**
 * @brief ユーザーが登録できるネットワーク接続設定の最大数です。
 */
static const int32_t UserNetworkProfileCountMax = 32;

/**
* @brief SSID リスト由来のネットワーク接続設定の最大数です。
*/
static const int32_t SsidListNetworkProfileCountMax = 300;

/**
* @brief ネットワーク接続設定の種類を表します。
*/
enum NetworkProfileType
{
    NetworkProfileType_User      = 1 << 0,   //!< ユーザー接続設定
    NetworkProfileType_SsidList  = 1 << 1,   //!< SSID リスト接続設定
    NetworkProfileType_Temporary = 1 << 2,   //!< テンポラリ接続設定
};

/**
 * @brief ネットワーク接続設定の基本情報を表します。
 */
struct NetworkProfileBasicInfo
{
    static const size_t NameSize = 64;          //!< 接続設定名の最大サイズ（バイト単位）
    nn::util::Uuid id;                          //!< 接続設定 ID
    char name[NameSize];                        //!< 接続設定名（UTF-8）
    NetworkProfileType networkProfileType;      //!< 接続設定の種類
    NetworkInterfaceType networkInterfaceType;  //!< リンク層の種類
    Ssid ssid;                                  //!< SSID
    Authentication authentication;              //!< 認証方式
    Encryption encryption;                      //!< 暗号化方式
};

/**
 * @brief ネットワーク接続設定を表します。
 */
struct NetworkProfileData
{
    nn::util::Uuid id;                             //!< 接続設定 ID
    char name[NetworkProfileBasicInfo::NameSize];  //!< 接続設定名（UTF-8）
    NetworkProfileType networkProfileType;         //!< 接続設定の種類
    NetworkInterfaceType networkInterfaceType;     //!< リンク層の種類
    bool isAutoConnect;                            //!< 利用要求提出時、この接続設定に対応する接続先（無線・有線）に自動的な接続を行うか否かのフラグ
    bool isLargeCapacity;                          //!< 大容量通信を許可するか否かのフラグ
    union
    {
        WirelessSettingData wirelessSetting;       //!< 無線 LAN 設定情報
        //EthernetSettingType ethernetSetting;
    };
    IpSettingData ipSetting;                       //!< IP 設定情報
};

/**
 * @brief SSID リストバージョンを表します。
 */
struct SsidListVersion
{
    static const size_t Size = 16;
    char data[Size];                   //!< SSID リストバージョンを表す null 終端された文字列
};

/**
 * @brief ネットワーク接続設定のハンドルです。
 */
struct NetworkProfileHandle
{
    void* _value;
};

inline bool operator==(const NetworkProfileData& lhs, const NetworkProfileData& rhs)
{
    return
        lhs.id == rhs.id &&
        nn::util::Strncmp(lhs.name, rhs.name, NetworkProfileBasicInfo::NameSize) == 0 &&
        lhs.networkProfileType == rhs.networkProfileType &&
        lhs.networkInterfaceType == rhs.networkInterfaceType &&
        lhs.isAutoConnect == rhs.isAutoConnect &&
        lhs.isLargeCapacity == rhs.isLargeCapacity &&
        lhs.wirelessSetting == rhs.wirelessSetting &&
        lhs.ipSetting == rhs.ipSetting;
}

inline bool operator!=(const NetworkProfileData& lhs, const NetworkProfileData& rhs)
{
    return !(lhs == rhs);
}

}
}
