﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

namespace nn
{
namespace nifm
{

/**
 * @brief   無効な ClientId の内部値です。
 */
static const uint32_t InvalidClientIdValue = 0;

/**
 * @brief ネットワーク接続設定ライブラリの利用者に割り当てられる識別子を表します。
 */
struct ClientId
{
    uint32_t value;

    //! @name 比較演算子
    //! @{

    /**
    * @brief   nn::nifm::ClientId 同士の等価演算子のオーバーロードです。
    *
    * @param[in]   lhs     比較するクライアント識別子の一方を指定します。
    * @param[in]   rhs     比較するクライアント識別子の他方を指定します。
    *
    * @return  両者が同じクライアントを指していれば true を、異なるクライアントを指していれば false を返します。
    *
    * @details
    */
    friend bool operator==(const ClientId& lhs, const ClientId& rhs) NN_NOEXCEPT
    {
        return lhs.value == rhs.value;
    }

    /**
    * @brief   nn::nifm::ClientId 同士の不等価演算子のオーバーロードです。
    *
    * @param[in]   lhs     比較するクライアント識別子の一方を指定します。
    * @param[in]   rhs     比較するクライアント識別子の他方を指定します。
    *
    * @return  両者が異なるクライアントを指していれば true を、同じクライアントを指していれば false を返します。
    *
    * @details
    */
    friend bool operator!=(const ClientId& lhs, const ClientId& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    bool IsValid() const NN_NOEXCEPT
    {
        return value != InvalidClientIdValue;
    }

    //! @}
};

}
}
