﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesRequirement.h>
#include <nn/util/util_Uuid.h>


namespace nn
{
namespace nifm
{
    //! @name 利用要求の要件設定・取得 API
    //! @{

    /**
     * @brief 利用要求の内容をプリセットで設定します。
     *
     * @param[in] handle            利用要求ハンドル
     * @param[in] requirementPreset 利用要求プリセット
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     *
     */
    nn::Result SetRequestRequirementPreset(RequestHandle handle, RequirementPreset requirementPreset) NN_NOEXCEPT;

    /**
     * @brief 提出済み利用要求をコピー元として指定し、利用要求の内容を設定します。
     *
     * @param[in] handle    利用要求ハンドル
     * @param[in] submitId  コピー元利用要求を表す提出番号
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     */
    nn::Result SetRequestRequirementBySubmitId(RequestHandle handle, uint32_t submitId) NN_NOEXCEPT;

    /**
     * @brief 利用要求の優先度を設定します。
     *
     * @param[in] handle    利用要求ハンドル
     * @param[in] priority  優先度
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *   - nn::nifm::UserRequestPriorityTop <= priority <= nn::nifm::UserRequestPriorityBottom
     *
     * @details
     *  優先度は、数値が小さいほど高い優先度を表します。
     */
    nn::Result SetRequestPriority(RequestHandle handle, uint8_t priority) NN_NOEXCEPT;

    /**
     * @brief 利用要求で利用する接続設定を指定します。
     *
     * @param[in] handle  利用要求ハンドル
     * @param[in] id      接続設定 ID
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     */
    nn::Result SetRequestNetworkProfileId(RequestHandle handle, const nn::util::Uuid& id) NN_NOEXCEPT;

    /**
     * @brief インターネット疎通確認に関する要求を設定します。
     *
     * @param[in] handle                        利用要求ハンドル
     * @param[in] connectionConfirmationOption  インターネット疎通確認に関する要求オプション
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     */
    nn::Result SetRequestConnectionConfirmationOption(RequestHandle handle, ConnectionConfirmationOption connectionConfirmationOption) NN_NOEXCEPT;

    /**
     * @brief 利用要求が受理されない場合に、拒否ではなく再度審査待ち状態に遷移するか否かを指定します。
     *
     * @param[in] handle        利用要求ハンドル
     * @param[in] isPersistent  継続的要求フラグ
     *
     * @return 処理の結果を表す nn::Result 値を返します。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     *  継続的要求フラグを true に設定した利用要求は、これが受理されない場合、却下されずに審査待ち状態に遷移します。
     *  このような利用要求を取り消すには、利用者が明示的に要求をキャンセルする必要があります。
     */
    nn::Result SetRequestPersistent(RequestHandle handle, bool isPersistent) NN_NOEXCEPT;

    //! @}
}
}

