﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesClientManagement.h>


namespace nn
{
namespace nifm
{
    //! @name クライアント管理 API
    //! @{

    /**
     * @brief   自分に割り当てられたクライアント識別子を取得します。
     *
     * @return  自分に割り当てられたクライアント識別子を返します。
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  自分に割り当てられた、ネットワーク接続管理ライブラリのクライアント識別子を取得します。
     */
    ClientId GetClientId() NN_NOEXCEPT;

    /**
     * @brief   指定したクライアントがインターネット接続を利用可能か確認します。
     *
     * @return  利用要求が受理されていれば true を、受理されていなければ false を返します。
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  指定したクライアントが提出した、インターネット接続の利用要求が受理されているかを確認します。@n
     */
    bool IsAnyInternetRequestAccepted(ClientId clientId) NN_NOEXCEPT;

    /**
     * @brief   ユーザーに意識される利用要求が受理されているかを確認します。
     *
     * @return  利用要求が受理されていれば true を、受理されていなければ false を返します。
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  ユーザーに意識される利用要求が受理されている状態で新たに利用要求を提出した場合、ユーザーが意識して利用している通信が切断される可能性があります。@n
     */
    bool IsAnyForegroundRequestAccepted() NN_NOEXCEPT;

    /**
     * @brief   バックグラウンドからのネットワーク利用要求が許可されるか否かを制御します。
     *
     * @param[in]   isEnabled    バックグラウンドからのネットワーク利用要求を許可するか否かを指定します。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  バックグラウンドからのネットワーク利用要求は、ユーザーに意識されるアプリやアプレット以外からの利用要求を意味します。
     *  このような利用要求は、デフォルト状態では許可されており、許可されなくなった場合、バックグラウンドの利用要求は却下されます。
     *  この設定は永続化されません。
     *
     */
    nn::Result SetBackgroundRequestEnabled(bool isEnabled) NN_NOEXCEPT;

    //! @}
}
}
