﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_IntrusiveList.h>

#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

class ISignalObject
{
public:
    ISignalObject() NN_NOEXCEPT
    {
    }
    virtual ~ISignalObject() NN_NOEXCEPT
    {
    }

    virtual void Signal() NN_NOEXCEPT = 0;
};

class SignalListItem : public ISignalObject, public nn::util::IntrusiveListBaseNode<SignalListItem>
{
private:
    ISignalObject& m_SignalObject;

public:
    explicit SignalListItem(ISignalObject& signalObject) NN_NOEXCEPT
        : m_SignalObject(signalObject)
    {
    }
    virtual ~SignalListItem() NN_NOEXCEPT
    {
    }

    virtual void Signal() NN_NOEXCEPT NN_OVERRIDE
    {
        m_SignalObject.Signal();
    }
};

class SignalList : public ISignalObject
{
private:
    nn::os::SdkMutex m_Mutex;
    nn::util::IntrusiveList<SignalListItem, nn::util::IntrusiveListBaseNodeTraits<SignalListItem>> m_SignalList;

public:
    SignalList() NN_NOEXCEPT
    {
    }
    virtual ~SignalList() NN_NOEXCEPT
    {
    }

    virtual void Signal() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        for (auto&& signalListItem : m_SignalList)
        {
            signalListItem.Signal();
        }
    }

    void Add(SignalListItem& signalListItem) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        m_SignalList.push_back(signalListItem);
    }

    bool Remove(SignalListItem& signalListItem) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        for (auto i = m_SignalList.begin(); i != m_SignalList.end(); ++i)
        {
            if (&*i == &signalListItem)
            {
                m_SignalList.erase(i);

                return true;
            }
        }

        return false;
    }
};

}
}
}
