﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn
{
namespace nifm
{
namespace detail
{

// コールバックの関数オブジェクトを表現するための基底クラスです
class CallbackObject
{
private:
    CallbackObject* m_pFollowingCallbackObject;

public:
    CallbackObject() NN_NOEXCEPT
        : m_pFollowingCallbackObject(nullptr)
    {
    }

    virtual ~CallbackObject() NN_NOEXCEPT
    {
    }

    // コールバックが実行された際、続けて実行するコールバックを登録します
    // 登録したコールバックは、 ExecuteImpl のあとに実行されます
    // コールバックの登録は連鎖します
    // EventHandler に登録済みの CallbackObject に対して、 EventHandler 外からこの操作をおこなってはいけません
    void AddFollowingCallback(CallbackObject* pInCallbackObject) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pInCallbackObject);

        CallbackObject* pCallback = this;
        NN_SDK_ASSERT(pCallback != pInCallbackObject, "A circular reference is detected in the callback succession.\n");

        while (pCallback->m_pFollowingCallbackObject != nullptr)
        {
            NN_SDK_ASSERT(pCallback != pInCallbackObject, "A circular reference is detected in the callback succession.\n");
            pCallback = pCallback->m_pFollowingCallbackObject;
        }

        pCallback->m_pFollowingCallbackObject = pInCallbackObject;
    }

    // コールバックの連鎖から pInCallbackObject 以下を切り離します
    // 渡された pInCallbackObject を切り離すことができれば true を、連鎖の中になく何もしなかった場合は false を返します
    // EventHandler に登録済みの CallbackObject に対して、 EventHandler 外からこの操作をおこなってはいけません
    bool RemoveCallback(const CallbackObject* pInCallbackObject) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pInCallbackObject);

        CallbackObject* pCallback = this;

        do
        {
            if (pCallback->m_pFollowingCallbackObject == pInCallbackObject)
            {
                pCallback->m_pFollowingCallbackObject = nullptr;
                return true;
            }

            pCallback = pCallback->m_pFollowingCallbackObject;
        } while (pCallback != nullptr);

        return false;
    }

    // 派生クラスで定義するコールバックの実装です
    virtual void ExecuteImpl() NN_NOEXCEPT
    {
    }

    // コールバックを実行します
    void Execute() NN_NOEXCEPT
    {
        ExecuteImpl();

        if (m_pFollowingCallbackObject != nullptr)
        {
            m_pFollowingCallbackObject->Execute();
        }
    }

    // コールバックを実行します
    void operator()() NN_NOEXCEPT
    {
        Execute();
    }

    const CallbackObject* GetFollowingCallback() const NN_NOEXCEPT
    {
        return m_pFollowingCallbackObject;
    }
};


class MultiEventHandler;

// イベントとコールバックを統合的に扱うためのインターフェースクラスです
class EventHandler : public nn::util::IntrusiveListBaseNode<EventHandler, EventHandler>
{
    friend class MultiEventHandler;

private:
    MultiEventHandler* m_pParentMultiEventHandler;

protected:
    CallbackObject m_CallbackHead;
    nn::os::MultiWaitType* m_pMultiWait;

public:
    EventHandler() NN_NOEXCEPT
        : m_pParentMultiEventHandler(nullptr),
          m_pMultiWait(nullptr)
    {
    }

    virtual ~EventHandler() NN_NOEXCEPT
    {
    }

    // イベントがシグナルしたときに実行するコールバックを登録します
    // MultiEventHandler に格納された状態で呼ぶとアサーションに失敗します
    void Register(CallbackObject* pInCallbackObject) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsRoot());
        RegisterInternal(pInCallbackObject);
    }

    // 登録済みのコールバックを解除します
    // 渡された pInCallbackObject を切り離すことができれば true を、連鎖の中になく何もしなかった場合は false を返します
    // MultiEventHandler に格納された状態で呼ぶとアサーションに失敗します
    bool Unregister(const CallbackObject* pInCallbackObject) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsRoot());

        if (pInCallbackObject == nullptr)
        {
            return false;
        }

        return UnregisterInternal(pInCallbackObject);
    }

    // イベントを多重待ちオブジェクトに登録します
    // nullptr を渡すと登録を解除します
    // すでに登録済みのところへ、さらに多重待ちオブジェクトを登録しようとするとアサーションに失敗します
    // MultiEventHandler に格納された状態で呼ぶとアサーションに失敗します
    void LinkTo(nn::os::MultiWaitType* pInMultiWait) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsRoot());
        LinkToInternal(pInMultiWait);
    }

    // 多重待ちの結果と比較し、シグナルしたのが自分のイベントであればシグナルを落としてコールバックを実行します
    // コールバックが実行された場合は true を、実行されなかった場合は false を返します
    // MultiEventHandler に格納された状態で呼ぶとアサーションに失敗します
    bool ExecuteCallbackIfSignaled(const nn::os::MultiWaitHolderType* pInMultiWaitHolder) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsRoot());
        return ExecuteCallbackIfSignaledInternal(pInMultiWaitHolder);
    }

    bool IsLinkedToMultiWait() const NN_NOEXCEPT
    {
        return m_pMultiWait != nullptr;
    }

    bool IsRoot() const NN_NOEXCEPT
    {
        return m_pParentMultiEventHandler == nullptr;
    }

private:
    // イベントがシグナルしたときに実行するコールバックを登録します
    void RegisterInternal(CallbackObject* pInCallbackObject) NN_NOEXCEPT
    {
        m_CallbackHead.AddFollowingCallback(pInCallbackObject);
    }

    // 登録済みのコールバックを解除します
    bool UnregisterInternal(const CallbackObject* pInCallbackObject) NN_NOEXCEPT
    {
        return m_CallbackHead.RemoveCallback(pInCallbackObject);
    }

    // イベントを多重待ちオブジェクトに登録します
    // nullptr を渡すと登録を解除します
    // すでに登録済みのところへ、さらに多重待ちオブジェクトを登録しようとするとアサーションに失敗します
    virtual void LinkToInternal(nn::os::MultiWaitType* pInMultiWait) NN_NOEXCEPT = 0;

    // 多重待ちの結果と比較し、シグナルしたのが自分のイベントであればシグナルを落としてコールバックを実行します
    // コールバックが実行された場合は true を、実行されなかった場合は false を返します
    virtual bool ExecuteCallbackIfSignaledInternal(const nn::os::MultiWaitHolderType* pInMultiWaitHolder) NN_NOEXCEPT = 0;

    void SetParent(MultiEventHandler* pInMultiEventHandler) NN_NOEXCEPT
    {
        m_pParentMultiEventHandler = pInMultiEventHandler;
    }
};


// イベントとコールバックを統合的に扱うための基底クラスです
// イベントの種類ごとに派生クラスが用意されます
class SingleGeneralEventHandlerBase : public EventHandler
{
protected:
    nn::os::MultiWaitHolderType m_MultiWaitHolder;  // 派生クラスで初期化する

    SingleGeneralEventHandlerBase() NN_NOEXCEPT
    {
    }

public:
    virtual ~SingleGeneralEventHandlerBase() NN_NOEXCEPT
    {
        LinkTo(nullptr);
        nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);
    }

    nn::os::MultiWaitHolderType* GetMultiWaitHolderPointer() NN_NOEXCEPT
    {
        return &m_MultiWaitHolder;
    }

private:
    // イベントを多重待ちオブジェクトに登録します
    // nullptr を渡すと登録を解除します
    // すでに登録済みのところへ、さらに多重待ちオブジェクトを登録しようとするとアサーションに失敗します
    virtual void LinkToInternal(nn::os::MultiWaitType* pInMultiWait) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT(pInMultiWait == nullptr || m_pMultiWait == nullptr);

        if (pInMultiWait == nullptr)
        {
            if (m_pMultiWait != nullptr)
            {
                nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
            }
        }
        else
        {
            nn::os::LinkMultiWaitHolder(pInMultiWait, &m_MultiWaitHolder);
        }
        m_pMultiWait = pInMultiWait;
    }

    // 多重待ちの結果と比較し、シグナルしたのが自分のイベントであればシグナルを落としてコールバックを実行します
    // コールバックが実行された場合は true を、実行されなかった場合は false を返します
    virtual bool ExecuteCallbackIfSignaledInternal(const nn::os::MultiWaitHolderType* pInMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE
    {
        if (pInMultiWaitHolder != &m_MultiWaitHolder)
        {
            return false;
        }

        if (!TryWait())
        {
            return false;
        }

        m_CallbackHead.Execute();

        return true;
    }

    virtual bool TryWait() NN_NOEXCEPT = 0;
    virtual void Clear() NN_NOEXCEPT = 0;
};


template<class T>
class SingleGeneralEventHandler : public SingleGeneralEventHandlerBase
{
private:
    T& m_Event;

    virtual bool TryWait() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Event.TryWait();
    }

    virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Event.Clear();
    }

public:
    explicit SingleGeneralEventHandler(T& event) NN_NOEXCEPT
        : m_Event(event)
    {
        nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, event.GetBase());
        nn::os::SetMultiWaitHolderUserData(&m_MultiWaitHolder, 0);
    }
};

using SingleEventHandler = SingleGeneralEventHandler<nn::os::Event>;
using SingleSystemEventHandler = SingleGeneralEventHandler<nn::os::SystemEvent>;
using SingleTimerEventHandler = SingleGeneralEventHandler<nn::os::TimerEvent>;


// 複数のイベントとコールバックを統合的に扱うためのクラスです
// 個々のイベントはまず Single*EventHandler に格納したうえで MultiEventHandler に登録します
class MultiEventHandler : public EventHandler
{
private:
    nn::util::IntrusiveList<EventHandler, nn::util::IntrusiveListBaseNodeTraits<EventHandler, EventHandler>> m_ChildEventHandlerList;

public:
    MultiEventHandler() NN_NOEXCEPT
    {
    }

    virtual ~MultiEventHandler() NN_NOEXCEPT NN_OVERRIDE
    {
        RemoveAll();
    }

    // EventHandler を登録します
    // 登録する EventHandler にすでに多重待ちオブジェクトが登録されている場合、アサーションに失敗します
    void Add(EventHandler& eventHandler) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!eventHandler.IsLinkedToMultiWait());

        m_ChildEventHandlerList.push_back(eventHandler);
        eventHandler.SetParent(this);
        eventHandler.RegisterInternal(&m_CallbackHead);

        if (m_pMultiWait != nullptr)
        {
            eventHandler.LinkToInternal(m_pMultiWait);
        }
    }

    // EventHandler の登録を解除します
    // EventHandler に登録された多重待ちオブジェクトも登録を解除されます
    void Remove(EventHandler& eventHandler) NN_NOEXCEPT
    {
        for (auto itr = m_ChildEventHandlerList.cbegin(); itr != m_ChildEventHandlerList.cend(); ++itr)
        {
            if (&(*itr) == &eventHandler)
            {
                m_ChildEventHandlerList.erase(itr);
                eventHandler.UnregisterInternal(&m_CallbackHead);
                eventHandler.SetParent(nullptr);
                eventHandler.LinkToInternal(nullptr);
                break;
            }
        }
    }

    // EventHandler の登録をすべて解除します
    // EventHandler に登録された多重待ちオブジェクトも登録を解除されます
    void RemoveAll() NN_NOEXCEPT
    {
        while (!m_ChildEventHandlerList.empty())
        {
            EventHandler& eventHandler = *m_ChildEventHandlerList.begin();
            Remove(eventHandler);
        }
    }

private:
    // イベントを多重待ちオブジェクトに登録します
    // nullptr を渡すと登録を解除します
    // すでに登録済みのところへ、さらに多重待ちオブジェクトを登録しようとするとアサーションに失敗します
    virtual void LinkToInternal(nn::os::MultiWaitType* pInMultiWait) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT(pInMultiWait == nullptr || m_pMultiWait == nullptr);

        for (auto&& childEventHandler : m_ChildEventHandlerList)
        {
            childEventHandler.LinkToInternal(pInMultiWait);   // MultiEventHandler に登録された状態で登録するので Internal を使う
        }

        m_pMultiWait = pInMultiWait;
    }

    // 多重待ちの結果と比較し、シグナルしたのが自分のイベントであればシグナルを落としてコールバックを実行します
    // コールバックが実行された場合は true を、実行されなかった場合は false を返します
    virtual bool ExecuteCallbackIfSignaledInternal(const nn::os::MultiWaitHolderType* pInMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE
    {
        for (auto&& childEventHandler : m_ChildEventHandlerList)
        {
            if (childEventHandler.ExecuteCallbackIfSignaledInternal(pInMultiWaitHolder))  // MultiEventHandler に登録された状態で実行するので Internal を使う
            {
                return true;
            }
        }

        return false;
    }
};

}
}
}
