﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/nifm_TypesNetworkInterface.h>

#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_LockGuard.h>
#include <nn/os/os_SdkMutex.h>

#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

class CancelFlagHolder
{
public:
    struct ReasonFlag
    {
        // 優先度の高い理由が上
        typedef nn::util::BitFlagSet<8, ReasonFlag>::Flag<0>  ShutdownRequired;                          // シャットダウン
        typedef nn::util::BitFlagSet<8, ReasonFlag>::Flag<1>  SleepRequired;                             // スリープ
        typedef nn::util::BitFlagSet<8, ReasonFlag>::Flag<2>  EthernetNetworkInterfaceNoLongerAvailable; // 有線OFF（開発環境のみ）
        typedef nn::util::BitFlagSet<8, ReasonFlag>::Flag<3>  WirelessNetworkInterfaceNoLongerAvailable; // 無線OFF
        typedef nn::util::BitFlagSet<8, ReasonFlag>::Flag<4>  InternetNoLongerRequested;                 // インターネット通信要求なし
        typedef nn::util::BitFlagSet<8, ReasonFlag>::Flag<5>  NetworkProfileSpecified;                   // 接続設定ID指定
        typedef nn::util::BitFlagSet<8, ReasonFlag>::Flag<6>  ConnectionConfirmationNoLongerRequested;   // 疎通確認不要
    };

    typedef nn::util::BitFlagSet<8, ReasonFlag> ReasonFlagSet;

private:
    mutable nn::os::SdkMutex m_Mutex;
    ReasonFlagSet m_Reason;

public:
    CancelFlagHolder() NN_NOEXCEPT
    {
        m_Reason.Reset();
    }

    ~CancelFlagHolder() NN_NOEXCEPT
    {
    }

    template<typename T>
    void RequireCancel() NN_NOEXCEPT
    {
        NN_DETAIL_NIFM_INFO("Cancel required: %d\n", T::Index);
        NN_UTIL_LOCK_GUARD(m_Mutex);
        m_Reason.Set<T>();
    }

    Result ConfirmConnectionConfirmationAdmitted(NetworkInterfaceType networkInterfaceType) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false
            || networkInterfaceType == NetworkInterfaceType_Ethernet
            || networkInterfaceType == NetworkInterfaceType_Ieee80211);

        NN_UTIL_LOCK_GUARD(m_Mutex);

        if (m_Reason.Test<ReasonFlag::ShutdownRequired>())
        {
            NN_RESULT_THROW(ResultDeviceShutdown());
        }
        if (m_Reason.Test<ReasonFlag::SleepRequired>())
        {
            NN_RESULT_THROW(ResultDevicePutToSleep());
        }
        if (m_Reason.Test<ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>())
        {
            if (networkInterfaceType == NetworkInterfaceType_Ethernet)
            {
                NN_RESULT_THROW(ResultNetworkInterfaceNoLongerAvailable());
            }
        }
        if (m_Reason.Test<ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>())
        {
            if (networkInterfaceType == NetworkInterfaceType_Ieee80211)
            {
                NN_RESULT_THROW(ResultNetworkInterfaceNoLongerAvailable());
            }
        }
        if (m_Reason.Test<ReasonFlag::InternetNoLongerRequested>())
        {
            NN_RESULT_THROW(ResultAggregatedRequestModified());
        }
        if (m_Reason.Test<ReasonFlag::NetworkProfileSpecified>())
        {
            NN_RESULT_THROW(ResultAggregatedRequestModified());
        }
        if (m_Reason.Test<ReasonFlag::ConnectionConfirmationNoLongerRequested>())
        {
            NN_RESULT_THROW(ResultAggregatedRequestModified());
        }

        NN_SDK_ASSERT(false
            || m_Reason.IsAllOff()
            || (m_Reason.Test<ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>() &&
                !m_Reason.Test<ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>())
            || (!m_Reason.Test<ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>() &&
                m_Reason.Test<ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>())
        );

        NN_RESULT_SUCCESS;
    }

    Result ConfirmInternetAdmitted(NetworkInterfaceType networkInterfaceType) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false
            || networkInterfaceType == NetworkInterfaceType_Ethernet
            || networkInterfaceType == NetworkInterfaceType_Ieee80211);

        NN_UTIL_LOCK_GUARD(m_Mutex);

        if (m_Reason.Test<ReasonFlag::ShutdownRequired>())
        {
            NN_RESULT_THROW(ResultDeviceShutdown());
        }
        if (m_Reason.Test<ReasonFlag::SleepRequired>())
        {
            NN_RESULT_THROW(ResultDevicePutToSleep());
        }
        if (m_Reason.Test<ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>())
        {
            if (networkInterfaceType == NetworkInterfaceType_Ethernet)
            {
                NN_RESULT_THROW(ResultNetworkInterfaceNoLongerAvailable());
            }
        }
        if (m_Reason.Test<ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>())
        {
            if (networkInterfaceType == NetworkInterfaceType_Ieee80211)
            {
                NN_RESULT_THROW(ResultNetworkInterfaceNoLongerAvailable());
            }
        }
        if (m_Reason.Test<ReasonFlag::InternetNoLongerRequested>())
        {
            NN_RESULT_THROW(ResultAggregatedRequestModified());
        }
        if (m_Reason.Test<ReasonFlag::NetworkProfileSpecified>())
        {
            NN_RESULT_THROW(ResultAggregatedRequestModified());
        }
        if (m_Reason.Test<ReasonFlag::ConnectionConfirmationNoLongerRequested>())
        {
            NN_RESULT_SUCCESS;
        }

        NN_SDK_ASSERT(false
            || m_Reason.IsAllOff()
            || (m_Reason.Test<ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>() &&
                !m_Reason.Test<ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>())
            || (!m_Reason.Test<ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>() &&
                m_Reason.Test<ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>())
        );

        NN_RESULT_SUCCESS;
    }

    Result ConfirmScanAdmitted() const NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        if (m_Reason.Test<ReasonFlag::ShutdownRequired>())
        {
            NN_RESULT_THROW(ResultDeviceShutdown());
        }
        if (m_Reason.Test<ReasonFlag::SleepRequired>())
        {
            NN_RESULT_THROW(ResultDevicePutToSleep());
        }
        if (m_Reason.Test<ReasonFlag::WirelessNetworkInterfaceNoLongerAvailable>())
        {
            NN_RESULT_THROW(ResultNetworkInterfaceNoLongerAvailable());
        }
        if (m_Reason.Test<ReasonFlag::InternetNoLongerRequested>())
        {
            NN_RESULT_THROW(ResultLowPriority());
        }
        if (m_Reason.Test<ReasonFlag::NetworkProfileSpecified>())
        {
            NN_RESULT_SUCCESS;
        }
        if (m_Reason.Test<ReasonFlag::ConnectionConfirmationNoLongerRequested>())
        {
            NN_RESULT_SUCCESS;
        }

        NN_SDK_ASSERT(false
            || m_Reason.IsAllOff()
            || m_Reason.Test<ReasonFlag::EthernetNetworkInterfaceNoLongerAvailable>()
        );

        NN_RESULT_SUCCESS;
    }

    void ClearCancel() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);
        NN_DETAIL_NIFM_INFO("Cancel Cleared: %d\n", m_Reason);
        m_Reason.Reset();
    }

    void ClearCancelByRequestModified() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        // InternetNoLongerRequested より優先度の低い理由をリセット
        m_Reason.Reset<ReasonFlag::InternetNoLongerRequested>();
        m_Reason.Reset<ReasonFlag::NetworkProfileSpecified>();
        m_Reason.Reset<ReasonFlag::ConnectionConfirmationNoLongerRequested>();
    }

    static CancelFlagHolder& GetSingleton() NN_NOEXCEPT;    // TODO: [TORIAEZU]
};

}
}
}
