﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/util/nifm_Heap.h>

#include <new>

namespace nn
{
namespace nifm
{
namespace detail
{

class ArrayedHeapBase
{
private:
    HeapBase& m_HeapBase;

    int m_Size;
    int m_Count;

protected:
    uintptr_t* m_pPointerList;

public:
    ArrayedHeapBase(HeapBase& heapBase, int size, uintptr_t* pPointerList) NN_NOEXCEPT
        : m_HeapBase(heapBase),
          m_Size(size),
          m_Count(0),
          m_pPointerList(pPointerList)
    {
        // m_HeapBase は派生クラスのデストラクタで構築されるため、ここで操作してはいけない

        for( int i = 0; i < m_Size; ++i )
        {
            m_pPointerList[i] = reinterpret_cast<uintptr_t>(nullptr);
        }
    }

    virtual ~ArrayedHeapBase() NN_NOEXCEPT
    {
        // m_HeapBase は派生クラスのデストラクタで破棄されるため、ここで操作してはいけない
    }

    // placement new からの利用を想定
    void* Allocate(size_t size) NN_NOEXCEPT
    {
        if( m_Count == m_Size )
        {
            return nullptr;
        }

        void* pBlock = m_HeapBase.Allocate(size);
        m_pPointerList[m_Count++] = reinterpret_cast<uintptr_t>(pBlock);
        return pBlock;
    }

    bool Free(void* pBlock) NN_NOEXCEPT
    {
        for( int i = 0; i < m_Count; ++i )
        {
            if( m_pPointerList[i] == reinterpret_cast<uintptr_t>(pBlock) )
            {
                return FreeByIndex(i);  // true になるはず
            }
        }

        return false;
    }

    bool FreeByIndex(int index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(index >= 0);

        if( index < m_Count )
        {
            void* pBlock = reinterpret_cast<void*>(m_pPointerList[index]);

            for( int i = index; i < m_Count - 1; ++i )
            {
                m_pPointerList[i] = m_pPointerList[i + 1];
            }

            m_pPointerList[--m_Count] = reinterpret_cast<uintptr_t>(nullptr);

            DestructObject(pBlock);
            m_HeapBase.Free(pBlock);

            return true;
        }

        return false;
    }

    void FreeAll()
    {
        while( GetCount() > 0 )
        {
            FreeByIndex(GetCount() - 1);
        }
    }

    int GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_Count;
    }

    bool Move(int fromIndex, int toIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(fromIndex >= 0);
        NN_SDK_ASSERT(toIndex >= 0);

        if( fromIndex > m_Count - 1 || toIndex > m_Count - 1 )
        {
            return false;
        }
        else if( fromIndex == toIndex)
        {
            return true;
        }

        uintptr_t ptr = m_pPointerList[fromIndex];

        if( fromIndex < toIndex )
        {
            for( int i = fromIndex; i < toIndex; ++i )
            {
                m_pPointerList[i] = m_pPointerList[i + 1];
            }
        }
        else
        {
            for( int i = fromIndex; i > toIndex; --i )
            {
                m_pPointerList[i] = m_pPointerList[i - 1];
            }
        }

        m_pPointerList[toIndex] = ptr;

        return true;
    }

private:
    // Free 時にデストラクタを呼ぶ場合は派生クラスでオーバーライドする
    virtual void DestructObject(void* pObject) NN_NOEXCEPT
    {
        NN_UNUSED(pObject);
    }
};

template<typename T>
class ArrayedFactoryBase : public ArrayedHeapBase
{
public:
    ArrayedFactoryBase(HeapBase& heapBase, int size, T** pPointerList) NN_NOEXCEPT
        : ArrayedHeapBase(heapBase, size, reinterpret_cast<uintptr_t*>(pPointerList))
    {
    }

    virtual ~ArrayedFactoryBase() NN_NOEXCEPT
    {
    }

    // operator new が nullptr を返すとコンストラクタを呼んでしまうコンパイラ対応
    template<typename... Ts>
    T* Create(Ts... args) NN_NOEXCEPT
    {
        void* p = Allocate(sizeof(T));

        if( p != nullptr )
        {
            return new(p) T(args...);
        }
        else
        {
            return nullptr;
        }
    }

    // using ArrayedHeapBase::Free; だとコンパイラ内部エラーになってしまうのでやむを得ず
    bool Free(void* pBlock) NN_NOEXCEPT
    {
        return ArrayedHeapBase::Free(pBlock);
    }

    // 引数は const 参照でも、内部でそのオブジェクトが破棄される可能性があることに注意
    bool Free(const T& object) NN_NOEXCEPT
    {
        for( int i = 0; i < GetCount(); ++i )
        {
            if( *reinterpret_cast<T*>(m_pPointerList[i]) == object )
            {
                return FreeByIndex(i);  // true になるはず
            }
        }

        return false;
    }

    T& operator[](int index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(index >= 0);

        return *reinterpret_cast<T*>(m_pPointerList[index]);
    }

    const T& operator[](int index) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(index >= 0);

        return *reinterpret_cast<T*>(m_pPointerList[index]);
    }

    T** begin() NN_NOEXCEPT
    {
        return reinterpret_cast<T**>(&m_pPointerList[0]);
    }

    const T* const * begin() const NN_NOEXCEPT
    {
        return reinterpret_cast<const T* const *>(&m_pPointerList[0]);
    }

    T** end() NN_NOEXCEPT
    {
        return reinterpret_cast<T**>(&m_pPointerList[GetCount()]);
    }

    const T* const * end() const NN_NOEXCEPT
    {
        return reinterpret_cast<const T* const *>(&m_pPointerList[GetCount()]);
    }

    bool MoveToTop(const T& object) NN_NOEXCEPT
    {
        int index;
        for( index = 0; index < GetCount(); ++index )
        {
            if( operator[](index) == object )
            {
                break;
            }
        }
        return Move( index, 0 );
    }

    bool MoveToEnd(const T& object) NN_NOEXCEPT
    {
        int index;
        for( index = 0; index < GetCount(); ++index )
        {
            if( operator[](index) == object )
            {
                break;
            }
        }
        return Move( index, GetCount() - 1 );
    }

private:
    // デストラクタを呼ぶ
    virtual void DestructObject(void* pObject) NN_NOEXCEPT
    {
        NN_UNUSED(pObject);     // TODO: T が POD 型のときに、これがないと pObject の未使用警告が出る？
        reinterpret_cast<T*>(pObject)->~T();
    }
};

template<typename T, int ListSize>
class ArrayedFactory : public ArrayedFactoryBase<T>
{
private:
    UnitHeap<T, ListSize> m_UnitHeap;
    T* m_PointerList[ListSize];

public:
    ArrayedFactory() NN_NOEXCEPT
        : ArrayedFactoryBase<T>(m_UnitHeap, ListSize, m_PointerList)
    {
    }

    virtual ~ArrayedFactory() NN_NOEXCEPT
    {
        // ヒープの実体がこのクラスで破棄されるので、ここでオブジェクトの解放処理をおこなう
        ArrayedHeapBase::FreeAll();
    }
};

}
}
}
