﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/service/nifm_Capability.h>
#include <nn/nifm/detail/service/nifm_INifmService.sfdl.h>
#include <nn/nifm/detail/util/nifm_SignalObject.h>
#include <nn/nifm/nifm_TypesClientManagement.h>

#include <nn/os/os_SystemEvent.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class RequestManager;
class UserRequest;

class RequestImpl
{
    NN_DISALLOW_COPY(RequestImpl);
    NN_DISALLOW_MOVE(RequestImpl);

private:
    RequestManager* m_pRequestManager;
    UserRequest* m_pUserRequest;
    Capability m_Capability;

    // 複数の Event をまとめてシグナルするためのラッパ
    class SystemEventSet : public ISignalObject
    {
    private:
        nn::os::SystemEvent m_SystemEvent1;
        nn::os::SystemEvent m_SystemEvent2;

    public:
        SystemEventSet() NN_NOEXCEPT;
        ~SystemEventSet() NN_NOEXCEPT;

        // 数字が若いほうが先にシグナルされることを保証する
        void Signal() NN_NOEXCEPT NN_OVERRIDE;

        nn::os::NativeHandle GetReadableHandle1() NN_NOEXCEPT;
        nn::os::NativeHandle GetReadableHandle2() NN_NOEXCEPT;
    } m_SystemEventSet;

public:
    RequestImpl(RequestManager* pRequestManager, const Capability& capability, RequirementPreset requirementPreset, ClientId clientId, nn::Bit64 processId) NN_NOEXCEPT;
    ~RequestImpl() NN_NOEXCEPT;

    bool IsAvailable() const NN_NOEXCEPT
    {
        return m_pUserRequest != nullptr;
    }

    nn::Result Submit() NN_NOEXCEPT;

    nn::Result Cancel() NN_NOEXCEPT;

    nn::Result GetRequestState(nn::sf::Out<IntRequestState> outRequestState) const NN_NOEXCEPT;

    nn::Result GetSystemEventReadableHandles(
        nn::sf::Out<nn::sf::NativeHandle> systemEventReadableHandle1,
        nn::sf::Out<nn::sf::NativeHandle> systemEventReadableHandle2) NN_NOEXCEPT;

    nn::Result GetResult() const NN_NOEXCEPT;

    nn::Result GetAdditionalInfo(nn::sf::Out<AdditionalInfo> outAdditionalInfo, nn::sf::Out<uint32_t> outRevision) const NN_NOEXCEPT;

    /**
     * @brief 利用要求の内容を直接設定します。
     */
    nn::Result SetRequirement(const Requirement& requirement) NN_NOEXCEPT;

    nn::Result GetRequirement(nn::sf::Out<Requirement> outRequirement) const NN_NOEXCEPT;

    nn::Result GetRevision(nn::sf::Out<uint32_t> outRevision) const NN_NOEXCEPT;

    /**
    * @brief 利用要求の内容をプリセットで設定します。
    */
    nn::Result SetRequirementPreset(int32_t requirementPreset) NN_NOEXCEPT;

    /**
    * @brief 利用要求の内容を指定された利用要求からコピーします。
    */
    nn::Result SetRequirementByRevision(uint32_t revision) NN_NOEXCEPT;

    /**
    * @brief 利用要求の優先度を設定します。
    */
    nn::Result SetPriority(uint8_t priority) NN_NOEXCEPT;

    /**
    * @brief 利用要求の優先度（直値）を設定します。
    */
    nn::Result SetRawPriority(uint8_t rawPriority) NN_NOEXCEPT;

    /**
     * @brief 利用要求で利用するプロファイルを設定します。
     */
    nn::Result SetNetworkProfileId(nn::util::Uuid uuid) NN_NOEXCEPT;

    /**
     * @brief 一度受理された利用要求がシステムに却下され得るか否かを指定します。
     */
    nn::Result SetRejectable(bool isRejectable) NN_NOEXCEPT;

    /**
     *@brief インターネット疎通確認に関する要求を設定します。
     */
    nn::Result SetConnectionConfirmationOption(int8_t connectionConfirmationOption) NN_NOEXCEPT;

    /**
     * @brief 利用要求の継続設定を指定します。
     */
    nn::Result SetPersistent(bool isPersistent) NN_NOEXCEPT;

    /**
     * @brief 利用要求の即時評価設定を指定します。
     */
    nn::Result SetInstant(bool isInstant) NN_NOEXCEPT;

    /**
     * @brief 利用要求の接続維持設定を指定します。
     */
    nn::Result SetSustainable(bool isSustainable, uint8_t priority) NN_NOEXCEPT;

    /**
     * @brief 利用要求が、帯域共有不可能な他の要求を排除してより大きい帯域を使用するか否か、を設定します。
     */
    nn::Result SetGreedy(bool isGreedy) NN_NOEXCEPT;

    /**
     * @brief 利用要求が、大きい帯域を要求する利用要求と帯域を共有可能であるか否か、を設定します。
     */
    nn::Result SetSharable(bool isSharable) NN_NOEXCEPT;

    /**
     * @brief 利用要求がスリープ遷移時にも取り下げずに維持するよう求めるか、を設定します。
     */
    nn::Result SetKeptInSleep(bool isKeptInSleep) NN_NOEXCEPT;

    /**
     * @brief スリープ中に維持するべきセッションの情報を登録します。
     */
    nn::Result RegisterSocketDescriptor(int32_t socketDescriptor) NN_NOEXCEPT;

    /**
     * @brief スリープ中に維持するべきセッションの情報を破棄します。
     */
    nn::Result UnregisterSocketDescriptor(int32_t socketDescriptor) NN_NOEXCEPT;

    /**
     * @brief エラーハンドリング用のアプレット起動に必要なパラメータを取得します。
     */
    nn::Result GetAppletInfo(
        nn::sf::Out<Bit32> outAppletId,
        nn::sf::Out<Bit32> outLibraryAppletMode,
        nn::sf::Out<uint32_t> outSize,
        nn::sf::OutBuffer outBuffer,
        uint32_t colorIndex) NN_NOEXCEPT;
};

}
}
}
