﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/core/nifm_NetworkResource.h>
#include <nn/nifm/detail/core/nifm_UserRequest.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/util/nifm_Heap.h>

#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_SdkMutex.h>

namespace nn
{
namespace nifm
{
namespace detail
{

class RequestManager
{
    NN_DISALLOW_COPY(RequestManager);
    NN_DISALLOW_MOVE(RequestManager);

public:
    enum class InternalRequestType : uint8_t
    {
        SustainConnection = 0,

        Count,
    };

    enum class State : uint8_t
    {
        FullAwake,
        MinimumAwake,

        ArrangingToSleep,
        ReadyToSleep,

        ArrangingToShutdown,
        ReadyToShutdown,

        Count
    };

private:
    static const size_t UserRequestCountMax = 32;

    UnitHeap<UserRequest, UserRequestCountMax> m_UserRequestHeap;

    mutable nn::os::SdkRecursiveMutex m_Mutex;
    nn::os::Event m_SubmitEvent;
    SingleEventHandler m_SubmitEventHandler;

    uint32_t m_LastRevision;    // リクエストが提出されるたびに更新されるリビジョン

    State m_State;
    ClientId m_ExclusiveClientId;
    bool m_IsBackgroundRequestEnabled;

    bool m_IsAggregatedRequestStaged;
    AggregatedRequestType m_AggregatedRequest;

private:
    nn::util::IntrusiveList<UserRequest, nn::util::IntrusiveListBaseNodeTraits<UserRequest>> m_RequestList;

    // TODO: 内部利用要求は一種類一個前提
    UserRequest* m_pInternalRequest[static_cast<int>(InternalRequestType::Count)];

    // 要求提出時に、利用要求を優先度順に並び替えます
    void Sort(const UserRequest* pInUserRequest) NN_NOEXCEPT;
    // 要求の受理が可能かを判断します
    // 要求の受理または拒否は呼び出し側でおこなう必要があります
    nn::Result SubmitCore(const UserRequest* pInUserRequest, UserRequest** ppOutInternalRequest) NN_NOEXCEPT;

    // 統合リクエスト作成の実装部分
    nn::Result AggregateSubmittedRequestsImpl(AggregatedRequestType* pOutAggregatedRequestType) NN_NOEXCEPT;

    // 指定された OnHold な要求が評価対象であるか否か
    bool IsEvaluandRequest(const UserRequest& userRequest) const NN_NOEXCEPT;

    // 指定された要求が特定の Internal Request か否か
    bool IsSpecifiedInternalRequest(const UserRequest& userRequest, InternalRequestType internetRequestType) const NN_NOEXCEPT;

    // RequestManager の状態が (Full/Minimum)Awake であるか
    bool IsAwake() const NN_NOEXCEPT;

    bool IsShuttingdown() const NN_NOEXCEPT;

    void ResetAllTemporaryExcludedFlags() NN_NOEXCEPT;

public:
    RequestManager() NN_NOEXCEPT;
    ~RequestManager() NN_NOEXCEPT;

    // ユーザーインターフェース側から利用される関数

    // 要求を生成し、リストに追加します。
    nn::Result CreateRequest(UserRequest** ppOutUserRequest, RequirementPreset requirementPreset, ISignalObject* pSignalObject, ClientId clientId, nn::Bit64 processId) NN_NOEXCEPT;

    // 内部利用要求を生成し、リストに追加します。
    nn::Result CreateInternalRequest(UserRequest** ppOutUserRequest, InternalRequestType requestType) NN_NOEXCEPT;

    // 要求をリストから削除し、破棄します。
    nn::Result DestroyRequest(UserRequest* pInUserRequest) NN_NOEXCEPT;

    // 要求を提出します。
    nn::Result Submit(UserRequest* pInUserRequest) NN_NOEXCEPT;

    // 要求を取り下げます。
    nn::Result Cancel(UserRequest* pInUserRequest) NN_NOEXCEPT;

    // 指定した ID のクライアントがインターネット接続の利用を許可されているかを確認します。
    bool IsAnyInternetRequestAccepted(ClientId clientId) NN_NOEXCEPT;


    // フォアグラウンド属性の利用要求が受理されているかを確認します。
    bool IsAnyForegroundRequestAccepted() NN_NOEXCEPT;

    // 評価対象の利用要求によって、無線から有線への自動切換が抑制されているかを確認します。
    bool IsAutomaticSwitchProhibited() NN_NOEXCEPT;

    // 利用要求の内容を取得します。
    nn::Result GetRequirement(Requirement *pOutRequirement, uint32_t revision) const NN_NOEXCEPT;


    // 接続試行モジュール側から利用される関数

    // 現在の統合リクエストを取得します。
    // 接続状態を変化させるために統合リクエストを参照する場合は、キャンセルの発生する StageAggregatedRequest() を使ってください。
    void GetAggregatedRequest(AggregatedRequestType* pOutAggregatedRequestType) const NN_NOEXCEPT;

    // 現在の統合リクエストを取得し、統合リクエストが大きく変化した場合にキャンセルが発生する状態へ遷移させます。
    void StageAggregatedRequest(AggregatedRequestType* pOutAggregatedRequestType) NN_NOEXCEPT;

    // 統合リクエストが大きく変化した場合にキャンセルが発生する状態を解除し、解除時点の統合リクエストを取得します。
    // 統合リクエストが不要な場合は pOutAggregatedRequestType に nullptr を渡すことができます。
    void UnstageAggregatedRequest(AggregatedRequestType* pOutAggregatedRequestType) NN_NOEXCEPT;

    // 現在提出されているユーザーリクエストに、ネットワークリソースの状態を反映します。
    // 接続状態が Available や Lost に変化したら、必ずこの関数を呼ぶ必要があります。
    // ネットワークリソースの確保に成功したリクエストは受領され、失敗したリクエストは却下されます。
    // TODO: 引数の統合
    nn::Result Renew(
        const TotalNetworkResourceInfo& totalNetworkResourceInfo,
        const AggregatedRequestType& aggregatedRequestType,
        const AdditionalInfo& additionalInfo,
        nn::Result result
    ) NN_NOEXCEPT;    // TODO: 名前

    void CancelSustainedRequests() NN_NOEXCEPT;

    //
    EventHandler& GetSubmitEventHandler() NN_NOEXCEPT
    {
        return m_SubmitEventHandler;
    }

    // スリープ中も維持するべきソケットの情報を取得します
    // そのとき受理されている利用要求に登録された情報だけが返ります
    int GetSocketInfoToKeepInSleep(SocketInfo* pOutSocketInfoArray, int count) const NN_NOEXCEPT;

    nn::Result PutToSleep(bool isPartial) NN_NOEXCEPT;
    nn::Result FullWakeUp() NN_NOEXCEPT;
    nn::Result MinimumWakeUp() NN_NOEXCEPT;
    nn::Result Shutdown() NN_NOEXCEPT;
    bool IsReadyToSleep() const NN_NOEXCEPT;
    bool IsReadyToShutdown() const NN_NOEXCEPT;
    bool IsFullAwaked() const NN_NOEXCEPT;

    nn::Result SetExclusiveClient(ClientId clientId) NN_NOEXCEPT;
    nn::Result SetBackgroundRequestEnabled(bool isEnabled) NN_NOEXCEPT;

    // 統合リクエストを更新し、必要ならキャンセルフラグを立てます。
    nn::Result UpdateAggregatedRequest() NN_NOEXCEPT;

    // すべてのリクエスト状態が Blocking などの遷移中状態ではないことを確認します
    bool IsStable() NN_NOEXCEPT;

    // for Debug
    void Dump() const NN_NOEXCEPT;

    // 外部疎通確認が不要なリクエストが受理されているかどうかを確認します
    bool IsProhibitedConnectionConfirmationOptionRequestAvailable() const NN_NOEXCEPT;
};

}
}
}

