﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/nifm_TypesInternetConnectionStatus.h>
#include <nn/util/util_Uuid.h>


namespace nn
{
namespace nifm
{
namespace detail
{

enum class NetworkResourceState
{
    Invalid,        // 不正、発生してはいけない
    Free,           // 新たに確保できる状態
    ToBePrepared,   // 確保処理中
    Available,      // 確保されて利用可能な状態
    Lost,           // 消失、 Free へ明示的な遷移が必要

    Count
};

struct NetworkResourceInfo
{
    NetworkType networkType;                            // ネットワーク種別
    NetworkResourceState networkResourceState;          // ネットワーク接続状態
    nn::Result connectionResult;                        // 接続の最新の結果（切断がブロックされた場合 Available でも切断理由が入ります）
    bool isDisconnectionBlocking;                       // 切断をブロックしているか

    InternetAvailability internetAvailability;          // インターネット接続（確認）状態
    nn::Result internetResult;                          // インターネット疎通確認の結果

    NetworkInterfaceType networkInterfaceType;          // ネットワークインターフェースの種類

    nn::util::Uuid networkProfileId;                    // 接続試行に使用された設定の ID

    uint32_t requestRevision;                           // 接続試行時の要求のリビジョン

    bool isGreedy;
    bool isSharable;
    uint32_t restrictedRevision;

    void Clear() NN_NOEXCEPT
    {
        networkType = NetworkType_None;
        networkResourceState = NetworkResourceState::Free;
        connectionResult = ResultNetworkNotFound();     // TODO
        isGreedy = false;
        isSharable = true;
        isDisconnectionBlocking = false;
        internetAvailability = InternetAvailability_Invalid;
        internetResult = ResultInternetConnectionNotAvailable();    // TODO
        networkInterfaceType = NetworkInterfaceType_Invalid;
        networkProfileId = nn::util::InvalidUuid;
        requestRevision = 0;
    }

    friend bool operator==(const NetworkResourceInfo& lhs, const NetworkResourceInfo& rhs) NN_NOEXCEPT
    {
        return lhs.networkType == rhs.networkType &&
               lhs.networkResourceState == rhs.networkResourceState &&
               lhs.connectionResult.GetInnerValueForDebug() == rhs.connectionResult.GetInnerValueForDebug() &&  // TODO: [TORIAEZU]
               lhs.isDisconnectionBlocking == rhs.isDisconnectionBlocking &&
               lhs.internetAvailability == rhs.internetAvailability &&
               lhs.internetResult.GetInnerValueForDebug() == rhs.internetResult.GetInnerValueForDebug() &&      // TODO: [TORIAEZU]
               lhs.networkInterfaceType == rhs.networkInterfaceType &&
               lhs.networkProfileId == rhs.networkProfileId &&
               lhs.requestRevision == rhs.requestRevision &&
               lhs.isGreedy == rhs.isGreedy &&
               lhs.isSharable == rhs.isSharable &&
               lhs.restrictedRevision == rhs.restrictedRevision;
    }

    friend bool operator!=(const NetworkResourceInfo& lhs, const NetworkResourceInfo& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

// TODO: 複数の NetworkResourceInfo を扱える独自の構造体／クラスに
// TODO: 名前
using TotalNetworkResourceInfo = NetworkResourceInfo;

}
}
}

