﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/nifm_TypesScan.h>
#include <nn/nifm/detail/core/networkInterface/nifm_NetworkInterfaceBase.h>
#include <nn/nifm/detail/core/accessPoint/nifm_WirelessAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_LocalAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_NeighborDetectionAccessPoint.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_MultipleWait.h>

#include <Wlanapi.h>

namespace nn
{
namespace nifm
{
namespace detail
{

// TODO:Windows に依存しないものは汎化
class WirelessInterface : public NetworkInterfaceBase
{
    NN_DISALLOW_COPY(WirelessInterface);
    NN_DISALLOW_MOVE(WirelessInterface);

    friend WirelessAccessPoint;
    friend LocalAccessPoint;
    friend NeighborDetectionAccessPoint;


public:
    // 現在の環境に接続された無線 NIC の MAC アドレスを列挙します。
    // Windows 環境ではホスト PC に接続されたすべての無線 NIC が対象になるため、
    // ターゲットデバイスとして利用できるより多くの NIC が列挙されることがあります。
    // pOutCount はすべての無線 NIC の数を返し、 inCount の値より大きくなることがあります。
    static void GetAllMacAddresses( MacAddress* pOutMacAddresses, int* pOutCount, int inCount ) NN_NOEXCEPT;

    //
    static nn::Result GetAllowedChannels(int16_t(&pOutChannels)[WirelessChannelsCountMax], int *pOutCount) NN_NOEXCEPT;

private:
    class AsyncTask
    {
    public:
        enum class Type
        {
            None,
            Scan,
            Connect,
            Disconnect,

            Count
        };

    private:
        Type m_Type;
        nn::os::Event m_CompleteEvent;
        DWORD m_WinApiError;
        nn::Result m_Result;

    public:
        AsyncTask() NN_NOEXCEPT
            : m_Type(Type::None),
              m_CompleteEvent(nn::os::EventClearMode_AutoClear),
              m_WinApiError(ERROR_SUCCESS)
        {
        }

        // for caller

        void Initialize(Type type) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Type == Type::None);
            bool tryWaitResult = m_CompleteEvent.TryWait();
            NN_SDK_ASSERT(!tryWaitResult);
            NN_UNUSED(tryWaitResult);

            m_Type = type;
        }

        void WaitForResponseOf(DWORD error) NN_NOEXCEPT
        {
            m_WinApiError = error;

            if( error == ERROR_SUCCESS )
            {
                m_CompleteEvent.Wait();
            }
            else
            {
                m_Type = Type::None;
                m_Result = ResultWindowsApiFailed();
            }
        }

        void WaitForResponseOf(DWORD error, const nn::TimeSpan& timeout) NN_NOEXCEPT
        {
            m_WinApiError = error;

            if( error == ERROR_SUCCESS )
            {
                if( !m_CompleteEvent.TimedWait(timeout) )
                {
                    m_Type = Type::None;
                    m_Result = ResultOperationTimedOut();     // TODO
                }
            }
            else
            {
                m_Type = Type::None;
                m_Result = ResultWindowsApiFailed();
            }
        }

        DWORD GetWinApiError() const NN_NOEXCEPT
        {
            return m_WinApiError;
        }

        nn::Result GetResult() const NN_NOEXCEPT
        {
            return m_Result;
        }

        // for callback

        Type GetType() const NN_NOEXCEPT
        {
            return m_Type;
        }

        void Complete(nn::Result result) NN_NOEXCEPT
        {
            m_Type = Type::None;
            m_Result = result;
            m_CompleteEvent.Signal();
        }
    };

private:
    bool m_IsInitialized;
    DWORD m_Index;

    ::HANDLE m_ClientHandle;
    GUID m_InterfaceGuid;

    MacAddress m_MacAddress;

    // TODO: TORIAEZU
    // 非同期処理用メンバ
    // すべての非同期処理（ブロッキング関数内処理）は以下を共有し、同時に一つしか走らない前提
    nn::os::SdkMutex m_BlockingTaskMutex;
    AsyncTask m_AsyncTask;

    int32_t m_Id;

    static const WCHAR profileBackupDirName[];

    union
    {
        nn::util::TypedStorage<WirelessAccessPoint, sizeof(WirelessAccessPoint), NN_ALIGNOF(WirelessAccessPoint)> m_WirelessAccessPointStorage;
        nn::util::TypedStorage<LocalAccessPoint, sizeof(LocalAccessPoint), NN_ALIGNOF(LocalAccessPoint)> m_LocalAccessPointStorage;
    } m_AccessPointStorage;

public:
    WirelessInterface() NN_NOEXCEPT;

    virtual ~WirelessInterface() NN_NOEXCEPT NN_OVERRIDE;

    nn::Result Initialize() NN_NOEXCEPT;
    nn::Result Initialize( const MacAddress& macAddress ) NN_NOEXCEPT;

    bool IsInitialized() const NN_NOEXCEPT;

    bool IsAvailable() const NN_NOEXCEPT;

    // 周囲をスキャンしてアクセスポイントなどの情報を更新します。
    // この関数はブロックします。
    nn::Result Scan() NN_NOEXCEPT;

    virtual NetworkInterfaceType GetNetworkInterfaceType() const NN_NOEXCEPT NN_OVERRIDE
    {
        return NetworkInterfaceType_Ieee80211;
    }

    virtual MacAddress* GetMacAddress( MacAddress* pOutMacAddress ) const NN_NOEXCEPT NN_OVERRIDE;

    // 周囲のアクセスポイントの一覧キャッシュを取得
    virtual nn::Result GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsLinkUp() const NN_NOEXCEPT NN_OVERRIDE;

private:
    virtual void* GetAccessPointStoragePointer() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_AccessPointStorage;
    }
    virtual size_t GetAccessPointStorageSize() NN_NOEXCEPT NN_OVERRIDE
    {
        return sizeof(m_AccessPointStorage);
    }

    nn::Result ConnectImpl(const WirelessAccessPoint& wirelessAccessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;
    nn::Result DisconnectImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT;
    nn::Result ReleaseImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPoint(WirelessAccessPoint* pInOutWirelessAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;

    nn::Result ConnectImpl(const LocalAccessPoint& localAccessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;
    nn::Result DisconnectImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT;
    nn::Result ReleaseImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPoint(LocalAccessPoint* pInOutLocalAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;

    nn::Result ConnectImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;
    nn::Result DisconnectImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT;
    nn::Result ReleaseImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPoint(NeighborDetectionAccessPoint* pInOutNeighborDetectionAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;

private:
    bool IsMediaConnected() const NN_NOEXCEPT;

    nn::Result WaitForIpChange( uint32_t timeoutMS ) NN_NOEXCEPT;

    void CopyCurrentProfileNameTo( WCHAR* profileName ) NN_NOEXCEPT;
    void ConnectByProfileName( WCHAR* profileName ) NN_NOEXCEPT;

    void SetWindowsNetworkProfileManualMode() const NN_NOEXCEPT;
    void RestoreWindowsNetworkProfiles() const NN_NOEXCEPT;

    virtual nn::Result UpdateRssiAndLinkLevel(AccessPointData* pInOutAccessPointData) const NN_NOEXCEPT NN_OVERRIDE;
    int ConvertLinkQualityToRssi(LONG linkQuality) const NN_NOEXCEPT;
    LinkLevel ConvertLinkQualityToLinkLevel(LONG linkQuality) const NN_NOEXCEPT;

private:
    static VOID WINAPI NotificationCallback(::PWLAN_NOTIFICATION_DATA pData, PVOID pContext) NN_NOEXCEPT;

    void NotificationCallbackImpl(::PWLAN_NOTIFICATION_DATA pData) NN_NOEXCEPT;

    virtual nn::Result GetScanChannels(int16_t* pOutScanChannels, int* pOutCount) const NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result PutToSleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result WakeUpImpl() NN_NOEXCEPT NN_OVERRIDE;
};

}
}
}

