﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/nifm_TypesScan.h>
#include <nn/nifm/detail/core/networkInterface/nifm_NetworkInterfaceBase.h>
#include <nn/nifm/detail/core/accessPoint/nifm_WirelessAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_LocalAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_NeighborDetectionAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>

#include <nn/nifm/detail/util/nifm_ArrayedFactory.h>

#include <nn/os/os_LightEvent.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_Optional.h>
#include <nn/wlan/wlan_Types.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class NetworkProfileManager;
class AccessPointListBase;

// TODO:Windows に依存しないものは汎化
class WirelessInterface : public NetworkInterfaceBase
{
    NN_DISALLOW_COPY(WirelessInterface);
    NN_DISALLOW_MOVE(WirelessInterface);

    friend WirelessAccessPoint;
    friend LocalAccessPoint;
    friend NeighborDetectionAccessPoint;

private:
    static const size_t WlanPassiveScanBufferSize                 = 32 * 1024;    // 100 ～ 110 個くらい (SIGLO-46820)
    static const size_t WlanActiveScanBufferSize                  =  4 * 1024;
    static const size_t PassivelyScannedAccessPointListBufferSize = 16 * 1024;    // 118個 (SIGLO-43820)
    static const size_t ActivelyScannedAccessPointListBufferSize  =  1 * 1024;
    static const int8_t RetryConnectCountMax                      = 3; // TORIAEZU
    static const int AccessPointHistoryCountMax                   = 2;
    static const int64_t ScanResultLifeTimeInSeconds              = 150;     // スキャン結果の有効期間

    class ScannedResultsHolder
    {
    private:
        AccessPointListBase& m_AccessPointList;
        bool m_IsValid;
        nn::TimeSpan m_Timestamp;

    public:
        explicit ScannedResultsHolder(AccessPointListBase& accessPointList) NN_NOEXCEPT
            : m_AccessPointList(accessPointList),
            m_IsValid(false),
            m_Timestamp(0)
        {
        }

        AccessPointListBase* GetAccessPointListPointer() NN_NOEXCEPT
        {
            return &m_AccessPointList;
        }

        const AccessPointListBase& GetAccessPointList() const NN_NOEXCEPT
        {
            return m_AccessPointList;
        }

        bool IsValid() const NN_NOEXCEPT
        {
            return m_IsValid;
        }

        void SetValid(bool isValid) NN_NOEXCEPT
        {
            m_IsValid = isValid;
        }

        void SetTimestamp(const nn::TimeSpan& timestamp) NN_NOEXCEPT
        {
            m_Timestamp = timestamp;
        }

        const nn::TimeSpan& GetTimestamp() const NN_NOEXCEPT
        {
            return m_Timestamp;
        }
    };

    class ScanRunner
    {
    private:
        WirelessInterface& m_WirelessInterface;

        const AccessPointData* m_pAccessPoint;
        char* m_pScanBuffer;
        size_t m_ScanBufferSize;

        static const size_t ThreadStackSize = 4 * 1024;
        char* m_pThreadStack;

        nn::os::ThreadType m_Thread;
        nn::os::LightEvent m_Event;
        nn::Result m_ScanResult;
        nn::Result m_CancelReasonResult;

    private:
        static void ThreadFunc(void* pContext) NN_NOEXCEPT;
        void ThreadFunc() NN_NOEXCEPT;

    public:
        ScanRunner(WirelessInterface& wirelessInterface, const AccessPointData* pAccessPoint, size_t scanBufferSize = WlanPassiveScanBufferSize) NN_NOEXCEPT;
        ~ScanRunner() NN_NOEXCEPT;

        nn::Result Start() NN_NOEXCEPT;
        nn::Result Cancel(nn::Result result) NN_NOEXCEPT;
        nn::Result Wait() NN_NOEXCEPT;

        nn::os::LightEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        nn::Result ExportTo(AccessPointListBase* pAccessPointList) const NN_NOEXCEPT;
    };

    class ConnectRunner
    {
    private:
        WirelessInterface& m_WirelessInterface;

        const WirelessAccessPoint& m_WirelessAccessPoint;
        const NetworkProfileBase& m_NetworkProfile;

        static const size_t ThreadStackSize = 4 * 1024;
        char* m_pThreadStack;

        nn::os::ThreadType m_Thread;
        nn::os::LightEvent m_Event;
        nn::Result m_ConnectResult;
        nn::Result m_CancelReasonResult;

    private:
        static void ThreadFunc(void* pContext) NN_NOEXCEPT;
        void ThreadFunc() NN_NOEXCEPT;

    public:
        ConnectRunner(WirelessInterface& wirelessInterface, const WirelessAccessPoint& wirelessAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;
        ~ConnectRunner() NN_NOEXCEPT;

        nn::Result Start() NN_NOEXCEPT;
        nn::Result Cancel(nn::Result result) NN_NOEXCEPT;
        nn::Result Wait() NN_NOEXCEPT;

        nn::os::LightEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }
    };

private:
    bool m_IsInitialized;

    MacAddress m_MacAddress;

    uint32_t m_OpenInfraModeCount;  // クラス全体での OpenInfraMode の呼出カウント
    bool m_IsCloseInfraModeNeeded;  // ReleaseImpl で Close する必要があるか = ConnectImpl 内で Open したか

    nn::os::LightEvent m_ScanCompleteEvent;

    class WlanConnectionEventCallback : public CallbackObject
    {
    private:
        WirelessInterface* m_pWirelessInterface;

    public:
        explicit WlanConnectionEventCallback(WirelessInterface* pWirelessInterface)
            : m_pWirelessInterface(pWirelessInterface)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE;
    };

    nn::util::optional<nn::os::SystemEvent> m_WlanConnectionEvent;
    nn::util::optional<WlanConnectionEventCallback> m_WlanConnectionEventCallback;
    nn::util::optional<SingleSystemEventHandler> m_WlanConnectionEventHandler;

    nn::Result m_ScanResult;
    nn::Result m_ConnectionAttemptResult;

    int32_t m_Id;
    mutable nn::os::SdkMutex m_Mutex;

    AccessPointList<FrameHeap<ActivelyScannedAccessPointListBufferSize>> m_ActivelyScannedAccessPointList;
    AccessPointList<ExpHeap<PassivelyScannedAccessPointListBufferSize>> m_PassivelyScannedAccessPointList;

    ScannedResultsHolder m_ActiveScanResults;
    ScannedResultsHolder m_PassiveScanResults;

    union
    {
        nn::util::TypedStorage<WirelessAccessPoint, sizeof(WirelessAccessPoint), NN_ALIGNOF(WirelessAccessPoint)> m_WirelessAccessPointStorage;
        nn::util::TypedStorage<LocalAccessPoint, sizeof(LocalAccessPoint), NN_ALIGNOF(LocalAccessPoint)> m_LocalAccessPointStorage;
        nn::util::TypedStorage<NeighborDetectionAccessPoint, sizeof(NeighborDetectionAccessPoint), NN_ALIGNOF(NeighborDetectionAccessPoint)> m_NeighborDetectionAccessPointStorage;
    } m_AccessPointStorage;

    ScannedResultsHolder* m_pLastScannedResults;

    int m_ActiveScanIndex;
    ArrayedFactory<AccessPointData, AccessPointHistoryCountMax> m_AccessPointHistory;

    nn::Result m_ResultHealth;

    int m_ScanChannelCount;
    int16_t m_ScanChannels[nn::wlan::WirelessChannelsCountMax];

public:
    // 現在の環境に接続された無線 NIC の MAC アドレスを列挙します。
    // pOutCount はすべての無線 NIC の数を返し、 inCount の値より大きくなることがあります。
    static void GetAllMacAddresses(MacAddress* pOutMacAddresses, int* pOutCount, int inCount) NN_NOEXCEPT;

    //
    static nn::Result GetAllowedChannels(int16_t(&pOutChannels)[WirelessChannelsCountMax], int *pOutCount) NN_NOEXCEPT;

public:
    WirelessInterface() NN_NOEXCEPT;

    virtual ~WirelessInterface() NN_NOEXCEPT NN_OVERRIDE;

    virtual NetworkInterfaceType GetNetworkInterfaceType() const NN_NOEXCEPT NN_OVERRIDE
    {
        return NetworkInterfaceType_Ieee80211;
    }

    virtual MacAddress* GetMacAddress(MacAddress* pOutMacAddress) const NN_NOEXCEPT NN_OVERRIDE;

    nn::Result Initialize() NN_NOEXCEPT;
    nn::Result Initialize(const MacAddress& macAddress) NN_NOEXCEPT;

    bool IsInitialized() const NN_NOEXCEPT;

    bool IsAvailable() const NN_NOEXCEPT;

    // 周囲をスキャンして無線アクセスポイントの情報を更新します。
    // この関数はブロックすることがあります。
    virtual nn::Result Scan() NN_NOEXCEPT NN_OVERRIDE;

    // 直近のパッシブスキャンの結果を取得します。
    virtual nn::Result GetScanData(AccessPointListBase* pOutAccessPointList) const NN_NOEXCEPT NN_OVERRIDE;

    // 保持するアクセスポイントの一覧を必要に応じて更新し、その結果を pOutAccessPointList に追加します
    // アクセスポイントの一覧の更新があれば true を、なければ false を返します。
    // 返り値が false になるまで呼び続けることで、アクセスポイントの一覧をもれなく受け取ることができます。
    // 繰り返し呼ぶと、先に接続履歴に基づいたアクティブスキャンをおこない、最後に全チャンネルのパッシブスキャンをおこないます
    virtual bool UpdateAccessPointList(AccessPointListBase* pOutAccessPointList) NN_NOEXCEPT NN_OVERRIDE;

    // 保持するアクセスポイントの一覧を pAccessPointList に追加します
    // 保持するアクセスポイントの一覧の直近の更新が timeSpan 以上前だった場合、 pAccessPointList には何も追加せず、
    // UpdateAccessPointList() の進捗を初期状態に戻します
    virtual nn::Result GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsLinkUp() const NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result ConfirmHealth() const NN_NOEXCEPT NN_OVERRIDE;

private:
    virtual void* GetAccessPointStoragePointer() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_AccessPointStorage;
    }
    virtual size_t GetAccessPointStorageSize() NN_NOEXCEPT NN_OVERRIDE
    {
        return sizeof(m_AccessPointStorage);
    }

    nn::Result ConnectImpl(const WirelessAccessPoint& wirelessAccessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;
    nn::Result DisconnectImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT;
    nn::Result ReleaseImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPoint(WirelessAccessPoint* pInOutWirelessAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;

    nn::Result ConnectImpl(const LocalAccessPoint& localAccessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;
    nn::Result DisconnectImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT;
    nn::Result ReleaseImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPoint(LocalAccessPoint* pInOutLocalAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;

    nn::Result ConnectImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;
    nn::Result DisconnectImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT;
    nn::Result ReleaseImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPoint(NeighborDetectionAccessPoint* pInOutNeighborDetectionAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;

    bool CopyToAccessPointListFromAccessPointData(AccessPointListBase* pOutAccessPointList, const AccessPointData& accessPoint) NN_NOEXCEPT;

    virtual nn::Result WisprAuthenticate(HttpResponse* pHttpResponse) NN_NOEXCEPT NN_OVERRIDE;

    // TSC ライブラリに渡すインターフェースの名前を返します
    virtual const char* GetInterfaceName() const NN_NOEXCEPT NN_OVERRIDE;

    // 無線デバイスのインフラ通信モードを開始／終了します。
    // 内部で参照カウントになっており、入れ子に呼ぶことができます。
    nn::Result OpenInfraMode() NN_NOEXCEPT;
    nn::Result CloseInfraMode() NN_NOEXCEPT;

    // ScanRunner から利用
    nn::Result ExecuteScan(void* pScanBuffer, size_t size, const AccessPointData* pInAccessPoint) NN_NOEXCEPT;
    nn::Result CancelScan() NN_NOEXCEPT;
    void InterpretScanData(AccessPointListBase* pOutAccessPointList, const void* pScanBuffer) NN_NOEXCEPT;

    // ConnectRunner から利用
    nn::Result ExecuteConnect(const AccessPointData& accessPoint, const NetworkProfileData& networkProfile) NN_NOEXCEPT;
    nn::Result CancelConnect() NN_NOEXCEPT;

    nn::Result Scan(AccessPointListBase* pOutAccessPointList, const AccessPointData* pAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPointHistory(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT;

    virtual nn::Result UpdateRssiAndLinkLevel(AccessPointData* pInOutAccessPointData) const NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result SetTcpSessionInformation(const SocketInfo& socketInfo) NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result SetScanChannels(const int16_t scanChannels[], int count) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result GetScanChannels(int16_t* pOutScanChannels, int* pOutCount) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result ClearScanChannels() NN_NOEXCEPT NN_OVERRIDE;

    virtual nn::Result PutToSleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result WakeUpImpl() NN_NOEXCEPT NN_OVERRIDE;
};

}
}
}

