﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesAccessPoint.h>
#include <nn/nifm/nifm_TypesRequestPrivate.h>
#include <nn/nifm/detail/core/nifm_NetworkResource.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/util/nifm_IpConfiguration.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_Optional.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class AccessPointBase;
class AccessPointListBase;
class HttpResponse;
class NetworkProfileBase;
struct NetworkResourceInfo;

// TODO:Windows に依存しないものは汎化
class NetworkInterfaceBase : public nn::util::IntrusiveListBaseNode<NetworkInterfaceBase>
{
    NN_DISALLOW_COPY(NetworkInterfaceBase);
    NN_DISALLOW_MOVE(NetworkInterfaceBase);

    friend AccessPointBase;

protected:
    static const MacAddress InvalidMacAddress;

private:
    uint32_t m_Id;

    mutable nn::os::SdkMutex m_NetworkResourceStateMutex;
    bool m_IsEnabled;
    NetworkType m_NetworkType;
    NetworkResourceState m_NetworkResourceState;
    uint32_t m_RequestRevision;
    AggregatedRequestType m_AggregatedRequestType;

    nn::os::Event m_ConnectionEvent;
    SingleEventHandler m_ConnectionSingleEventHandler;
    MultiEventHandler m_ConnectionEventHandler;

    class ConnectionEventCallback : public CallbackObject
    {
    private:
        NetworkInterfaceBase& m_NetworkInterface;

    public:
        explicit ConnectionEventCallback(NetworkInterfaceBase& networkInterface)
            : m_NetworkInterface(networkInterface)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            m_NetworkInterface.ConnectionEventCallback();
        }
    } m_ConnectionEventCallback;

    nn::util::optional<SingleSystemEventHandler> m_IpSingleEventHandler;

    class IpEventCallbackType : public CallbackObject
    {
    private:
        NetworkInterfaceBase& m_NetworkInterface;

    public:
        explicit IpEventCallbackType(NetworkInterfaceBase& networkInterface)
            : m_NetworkInterface(networkInterface)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            m_NetworkInterface.IpEventCallback();
        }
    };
    nn::util::optional<IpEventCallbackType> m_IpEventCallback;

    bool m_IsInternetConnectionInitialized;
    InternetAvailability m_InternetAvailability;
    nn::Result m_InternetResult;

    AdditionalInfo m_AdditionalInfo;

    bool m_IsDisconnectionBlocking;
    nn::Result m_ConnectionResult;

protected:
    AccessPointBase* m_pAccessPoint;
    NetworkProfileData m_NetworkProfileData;

    IpConfiguration m_IpConfiguration;

public:
    static const int InvalidScanChannelCount;

public:
    NetworkInterfaceBase() NN_NOEXCEPT;
    virtual ~NetworkInterfaceBase() NN_NOEXCEPT;

    virtual NetworkInterfaceType GetNetworkInterfaceType() const NN_NOEXCEPT = 0;

    virtual MacAddress* GetMacAddress( MacAddress* pOutMacAddress ) const NN_NOEXCEPT = 0;

    int32_t GetId() const NN_NOEXCEPT
    {
        return m_Id;
    }
    void SetId(int32_t id) NN_NOEXCEPT
    {
        m_Id = id;
    }

    // 周囲をスキャンしてアクセスポイントなどの情報を更新します。
    // この関数はブロックすることがあります。
    // 派生クラスで再定義しなければ何もしません。
    virtual nn::Result Scan() NN_NOEXCEPT;

    // 直近のスキャンの結果を取得します。
    // この関数が返す情報は、必ずしも GetLatestAccessPointList() の一覧とは一致しません
    // 派生クラスで再定義しなければ何もしません。
    virtual nn::Result GetScanData(AccessPointListBase* pOutAccessPointList) const NN_NOEXCEPT;

    // 保持するアクセスポイントの一覧を必要に応じて更新し、その結果を pOutAccessPointList に追加します
    // アクセスポイントの一覧の更新があれば true を、なければ false を返します。
    // 返り値が false になるまで呼び続けることで、アクセスポイントの一覧をもれなく受け取ることができます。
    // 派生クラスで再定義しなければ何もせず、常に false を返します。
    virtual bool UpdateAccessPointList(AccessPointListBase* pOutAccessPointList) NN_NOEXCEPT;

    // 保持するアクセスポイントの一覧を pAccessPointList に追加します
    // 保持するアクセスポイントの一覧の直近の更新が timeSpan 以上前だった場合、 pAccessPointList には何も追加せず、
    // UpdateAccessPointList() の進捗を初期状態に戻します
    virtual nn::Result GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT = 0;

    // 以下の接続操作 API は互いにスレッドアンセーフです

    // ネットワークリソース（接続等）の確保を試行します
    // ネットワークリソースが解放状態である必要があります
    // ネットワークリソース解放状態でこの関数を呼んだ場合、結果の成否にかかわらず、再度呼べるようにするには解放が必要になります
    nn::Result Connect(const AccessPointBase& accessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;

    // 現在確保しているネットワークリソース（接続等）を破棄します
    // 再度接続試行可能な状態になるには、破棄後、明示的な解放が必要です
    nn::Result Disconnect(nn::Result connectionResult) NN_NOEXCEPT;

    // 現在確保しているネットワークリソース（接続等）を解放します
    // 破棄されていない有効なネットワークリソースがある場合、解放に失敗します
    nn::Result Release() NN_NOEXCEPT;

    // 現在の接続そのままで新しい統合リクエストを受け入れられるか確認し、
    // 受け入れられれば保持している統合リクエストを更新します
    // 確認の過程で、インターネット接続状態の更新を試みることがあります
    nn::Result ConfirmKeepingCurrentConnection(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;

    // 有効な接続が存在する場合、使用中の接続設定を更新します
    nn::Result UpdateCurrentNetworkProfile(const NetworkProfileData& networkProfileData) NN_NOEXCEPT;

    // ネットワークインターフェースの有効／無効を返します
    bool IsEnabled() const NN_NOEXCEPT;

    // ネットワークインターフェースの有効／無効を切り替えます
    // ネットワークインターフェースが無効な場合、 Connect() が失敗を返します
    // ネットワークリソースが確保されている状態でネットワークインターフェースを無効化した場合、
    // ConnectionEvent がシグナルし、そのコールバックの中で Disconnect() が呼ばれます
    // ネットワークインターフェースの有効／無効ステータスは上記コールバックの実行を待たずに遷移します
    // この関数と Connect(), Disconnect(), Release() の各関数はスレッドセーフです
    nn::Result SetEnabled(bool isEnabled) NN_NOEXCEPT;


    // 現在確保しているネットワークリソース（接続等）の状態・情報を取得します
    // 状態は Free → ToBePrepared → (Available) → Lost → Free と変化します
    // Free → ToBePrepared の遷移は Connect 関数を呼ばなければ発生しません
    // Lost → Free の遷移は Release 関数を呼ばなければ発生しません
    void GetNetworkResourceInfo(NetworkResourceInfo* pOutNetworkResourceInfo) const NN_NOEXCEPT;

    // 接続中のアクセスポイントの情報を取得します
    nn::Result GetCurrentAccessPoint(AccessPointData* pOutAccessPointData) const NN_NOEXCEPT;

    // 利用中のネットワーク接続設定の情報を取得します
    nn::Result GetCurrentNetworkProfile(NetworkProfileData* pOutNetworkProfile) const NN_NOEXCEPT;

    virtual bool IsLinkUp() const NN_NOEXCEPT = 0;

    EventHandler& GetConnectionEventHandler() NN_NOEXCEPT
    {
        return m_ConnectionEventHandler;
    }

    nn::Result GetCurrentIpAddress( IpV4Address* pOutIpAddress ) const NN_NOEXCEPT;

    nn::Result GetCurrentIpConfigInfo( IpAddressSetting* pOutIpAddressSetting, DnsSetting* pOutDnsSetting ) const NN_NOEXCEPT;

    nn::Result GetRedirectInfoIfAny(AdditionalInfo* pOutAdditionalInfo) const NN_NOEXCEPT;

    // インターフェースの健康状態を確認します。
    // 低層が不正な状態になった場合に失敗を返すことがあります。
    // 派生クラスでオーバーライドしなければ常時成功（健康）を返します。
    virtual Result ConfirmHealth() const NN_NOEXCEPT;

    // スリープ中に維持するべきセッションの情報を登録します。
    // 未対応のインターフェースからは ResultNotImplemented が返ります。
    virtual nn::Result SetTcpSessionInformation(const SocketInfo& socketInfo) NN_NOEXCEPT;

    // ネットワークインタフェースにスキャン対象チャンネルを設定します。
    virtual nn::Result SetScanChannels(const int16_t scanChannels[], int count) NN_NOEXCEPT;
    virtual nn::Result GetScanChannels(int16_t* pOutScanChannels, int* pOutCount) const NN_NOEXCEPT;
    virtual nn::Result ClearScanChannels() NN_NOEXCEPT;

    nn::Result PutToSleep() NN_NOEXCEPT;

    nn::Result WakeUp() NN_NOEXCEPT;
    //bool IsReadyToSleep() const NN_NOEXCEPT;

protected:
    // ネットワークインターフェースが予期せずリンク層の接続を失ったことを基底クラスに通知します
    // DisconnectImpl の実装から呼び出す必要はありません
    void NotifyLinkDisconnected(nn::Result connectionResult) NN_NOEXCEPT;

    // LAN 内通信を含むインターネット通信を開始する際、 ConnectImpl から呼ばれる必要があります
    nn::Result InitializeInternetConnection(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;

    // 現在確保しているネットワークリソース（接続等）の状態を取得します
    NetworkResourceState GetNetworkResourceState() const NN_NOEXCEPT;

    // GetConnectionEventHandler() で取得される EventHandler に EventHandler を登録します
    void AddEventHandler(EventHandler& eventHandler) NN_NOEXCEPT;

    // GetConnectionEventHandler() で取得される EventHandler から EventHandler を解除します
    void RemoveEventHandler(EventHandler& eventHandler) NN_NOEXCEPT;

private:
    virtual void* GetAccessPointStoragePointer() NN_NOEXCEPT = 0;
    virtual size_t GetAccessPointStorageSize() NN_NOEXCEPT = 0;

    nn::Result Disconnect(const AccessPointBase* pAccessPoint, nn::Result connectionResult) NN_NOEXCEPT;
    nn::Result Release(const AccessPointBase* pAccessPoint) NN_NOEXCEPT;

    nn::Result ConnectionEventCallback() NN_NOEXCEPT;
    nn::Result IpEventCallback() NN_NOEXCEPT;

    // TSC ライブラリに渡すインターフェースの名前を返します
    virtual const char* GetInterfaceName() const NN_NOEXCEPT;

    // 現在の接続そのままで新しい統合リクエストを受け入れられるか確認します
    // 確認の過程で、インターネット接続状態の更新を試みることがあります
    nn::Result ConfirmAcceptability(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;

    // 必要であればインターネットの疎通を確認し、インターネット接続状態を更新します
    nn::Result UpdateInternetAvailability(
        ConnectionConfirmationOption connectionConfirmationOption,
        ConnectionConfirmationOption connectionConfirmationOptionSub) NN_NOEXCEPT;

    virtual nn::Result WisprAuthenticate(HttpResponse* pHttpResponse) NN_NOEXCEPT;

    // Release から呼ばれ、必要に応じて LAN 内通信を含むインターネット通信を終了します
    nn::Result FinalizeInternetConnectionIfNecessary() NN_NOEXCEPT;

    void SetStateAndSignal(NetworkResourceState state) NN_NOEXCEPT;

    // 派生クラスで再定義しなければ何もせず失敗を返します
    virtual nn::Result UpdateRssiAndLinkLevel(AccessPointData* pInOutAccessPointData) const NN_NOEXCEPT;

    virtual nn::Result PutToSleepImpl() NN_NOEXCEPT = 0;
    virtual nn::Result WakeUpImpl() NN_NOEXCEPT = 0;
};

}
}
}

