﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/detail/core/networkInterface/nifm_NetworkInterfaceBase.h>
#include <nn/nifm/detail/core/accessPoint/nifm_EthernetAccessPoint.h>

#include <nn/eth/eth_EthTypes.h>
#include <nn/eth/eth_EthClient.h>

#include <nn/nn_TimeSpan.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_TypedStorage.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class AccessPointListBase;
class EthernetNetworkProfile;

class EthernetInterface : public NetworkInterfaceBase, public nn::util::IntrusiveListBaseNode<EthernetInterface>
{
    NN_DISALLOW_COPY(EthernetInterface);
    NN_DISALLOW_MOVE(EthernetInterface);

    friend EthernetAccessPoint;

private:
    static const int WarmingTimeInSeconds = 8;  // オブジェクト生成時に LAN ケーブルがささっていたとして、リンクアップするまでに見込まれる時間

private:
    mutable nn::eth::client::InterfaceHandler m_InterfaceHandler;   // TODO: mutable をはずす

    nn::os::SystemEvent* m_pInterfaceEvent;
    class InterfaceEventCallback : public CallbackObject
    {
    private:
        EthernetInterface* m_pEthernetInterface;

    public:
        explicit InterfaceEventCallback(EthernetInterface* pEthernetInterface)
            : m_pEthernetInterface(pEthernetInterface)
        {
        }

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE;
    } m_InterfaceEventCallback;

    nn::util::optional<SingleSystemEventHandler> m_InterfaceEventHandler;

    nn::eth::MediaType m_MediaCurrent;  // EthernetInterface として認識している nn::eth::InterfaceHandler の現在の MediaType

    nn::TimeSpan m_ReadyTime;   // 接続直後のリンクアップが完了すると見込まれる時刻

    bool m_IsInitialized;

    union
    {
        nn::util::TypedStorage<EthernetAccessPoint, sizeof(EthernetAccessPoint), NN_ALIGNOF(EthernetAccessPoint)> m_EthernetAccessPointStorage;
    } m_AccessPointStorage;

public:
    EthernetInterface() NN_NOEXCEPT;

    virtual ~EthernetInterface() NN_NOEXCEPT NN_OVERRIDE;

    nn::Result Initialize(nn::eth::client::InterfaceGroupHandler* pInterfaceGroupHandler, nn::eth::InterfaceInfo* pInterfaceInfo) NN_NOEXCEPT;

    virtual NetworkInterfaceType GetNetworkInterfaceType() const NN_NOEXCEPT NN_OVERRIDE
    {
        return NetworkInterfaceType_Ethernet;
    }

    virtual MacAddress* GetMacAddress( MacAddress* pOutMacAddress ) const NN_NOEXCEPT NN_OVERRIDE;

    bool IsInitialized() const NN_NOEXCEPT;

    bool IsAvailable() const NN_NOEXCEPT;

    // 呼ばれた時点の Ethernet のリンクアップ状況を反映し、仮想的なアクセスポイントを pAccessPointList に追加します
    virtual nn::Result GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsLinkUp() const NN_NOEXCEPT NN_OVERRIDE;

    // USB Ethernet アダプターがまだ存在するかを返す
    // ネットワークリソースが解放されていないときは、実際には抜去されていてもまだ存在する扱いにする
    bool IsPresent() const NN_NOEXCEPT;

    EthernetInterface* Find(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT;

private:
    virtual void* GetAccessPointStoragePointer() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_AccessPointStorage;
    }
    virtual size_t GetAccessPointStorageSize() NN_NOEXCEPT NN_OVERRIDE
    {
        return sizeof(m_AccessPointStorage);
    }

    nn::Result ConnectImpl(const EthernetAccessPoint& ethernetAccessPoint, const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT;
    nn::Result DisconnectImpl(const EthernetAccessPoint& ethernetAccessPoint) NN_NOEXCEPT;
    nn::Result ReleaseImpl(const EthernetAccessPoint& ethernetAccessPoint) NN_NOEXCEPT;
    void UpdateAccessPoint(EthernetAccessPoint* pInOutEthernetAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT;

    // TSC ライブラリに渡すインターフェースの名前を返します
    virtual const char* GetInterfaceName() const NN_NOEXCEPT NN_OVERRIDE;

    // リンク状態変化時に呼ばれるコールバック
    void InterfaceEventCallback() NN_NOEXCEPT;

    virtual nn::Result PutToSleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result WakeUpImpl() NN_NOEXCEPT NN_OVERRIDE;
};

}
}
}
