﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/nifm_TypesAccessPoint.h>

#include <nn/util/util_IntrusiveList.h>
#include <nn/nn_TimeSpan.h>

namespace nn
{
namespace nifm
{
namespace detail
{

class NetworkInterfaceBase;
class NetworkProfileBase;

class WirelessAccessPoint;
class LocalAccessPoint;
class EthernetAccessPoint;

class AccessPointBase : public nn::util::IntrusiveListBaseNode<AccessPointBase>
{
    friend NetworkInterfaceBase;

protected:
    mutable NetworkInterfaceBase* m_pNetworkInterfaceBase;
    nn::TimeSpan m_Timestamp;

public:
    virtual ~AccessPointBase() NN_NOEXCEPT;

    bool IsPossiblyAvailable(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT;

    nn::Result Connect(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT;
    nn::Result Disconnect(nn::Result connectionResult) const NN_NOEXCEPT;
    nn::Result Release() const NN_NOEXCEPT;

    NetworkInterfaceBase& GetNetworkInterface() NN_NOEXCEPT
    {
        return *m_pNetworkInterfaceBase;
    }

    const NetworkInterfaceBase& GetNetworkInterface() const NN_NOEXCEPT
    {
        return *m_pNetworkInterfaceBase;
    }

    const nn::TimeSpan& GetTimestamp() const NN_NOEXCEPT
    {
        return m_Timestamp;
    }

    virtual NetworkType GetNetworkType() const NN_NOEXCEPT = 0;

    // ソート時の優先順位を返します
    // 値が小さいほど高優先度です
    // オーバーライドしない場合 0 を返します
    virtual int GetPriority() const NN_NOEXCEPT;

    // オブジェクトの内容を AccessPointData に変換します
    // 仮想的なアクセスポイントの場合、 失敗を返し、エクスポートをしません
    virtual nn::Result TryExport(AccessPointData* pOutAccessPointData) const NN_NOEXCEPT = 0;

    // オブジェクトを p で渡されたサイズ size のバッファにコピーします
    virtual AccessPointBase* CopyTo(void* p, size_t size) const NN_NOEXCEPT = 0;
    // オブジェクトのサイズを返します
    virtual size_t GetSize() const NN_NOEXCEPT = 0;

    virtual bool operator==(const AccessPointBase& accessPoint) const NN_NOEXCEPT = 0;  // 多重ディスパッチのために、派生クラスで左辺と右辺を入れ替えた比較演算でオーバーライドする必要があります
    virtual bool operator==(const WirelessAccessPoint& accessPoint) const NN_NOEXCEPT;  // オーバーライドしなければ常に false を返します
    virtual bool operator==(const LocalAccessPoint& accessPoint) const NN_NOEXCEPT;     // オーバーライドしなければ常に false を返します
    virtual bool operator==(const EthernetAccessPoint& accessPoint) const NN_NOEXCEPT;  // オーバーライドしなければ常に false を返します

    bool operator!=(const AccessPointBase& accessPoint) const NN_NOEXCEPT
    {
        return !(*this == accessPoint);
    }

private:
    virtual nn::Result ConnectImpl(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT = 0;
    virtual nn::Result DisconnectImpl() const NN_NOEXCEPT = 0;
    virtual nn::Result ReleaseImpl() const NN_NOEXCEPT = 0;
    virtual void UpdateAccessPoint(const NetworkProfileBase& networkProfile) NN_NOEXCEPT = 0;

    // 派生クラスが持つアトリビュートが指定された条件に合致するか調べます
    // 基底クラスが持つアトリビュートが指定された条件に合致しない場合、この関数は呼ばれません
    // 派生クラスが独自のアトリビュートを持たない場合、オーバーライドは不要です
    virtual bool IsPossiblyAvailableImpl(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT;

protected:
    explicit AccessPointBase(NetworkInterfaceBase* pNetworkInterfaceBase) NN_NOEXCEPT;
    explicit AccessPointBase(const AccessPointBase& accessPointBase) NN_NOEXCEPT;
};

}
}
}
