﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief      NFP(amiibo) に関する非公開 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nfp/nfp_Result.h>
#include <nn/nfp/nfp_PrivateResult.h>
#include <nn/nfp/nfp_PrivateTypes.h>

namespace nn {
namespace nfp {

/**
 * @name    Private API
 * @{
 */

/**
 * @brief   NFP ライブラリをシステムモードで初期化します。
 *
 * @pre
 *  - GetState() == @ref State_None
 *
 * @post
 *  - NFP ライブラリがシステムモードで初期化済みの状態になります。
 *  - GetState() == @ref State_Init
 *
 * @details
 * nfp_PrivateApi.h で定義されたシステム用の API を使用するためには、
 * この API で NFP ライブラリを初期化する必要があります。@n
 * システムモードでは、一般公開 API も使用できます。ただし、以下の API は除きます。@n
 * - StartNicknameAndOwnerSettings(DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, RegisterInfo* pOutRegisterInfo, const AmiiboSettingsStartParam& startParam, const TagInfo& tagInfo, const RegisterInfo& registerInfo)
 * - StartNicknameAndOwnerSettings(DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, RegisterInfo* pOutRegisterInfo, const AmiiboSettingsStartParam& startParam, const TagInfo& tagInfo)
 * - StartGameDataEraser(DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam, const TagInfo& tagInfo)
 * - StartRestorer(DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam, const TagInfo& tagInfo)
 * - OpenApplicationArea()
 * - GetApplicationArea()
 * - SetApplicationArea()
 * - RecreateApplicationArea()
 * - CreateApplicationArea()
 *
 */
void InitializeSystem() NN_NOEXCEPT;

/**
 * @brief   NFP ライブラリを終了します。
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @post
 *  - GetState() == @ref State_None
 *
 * @details
 *  この関数は NFP ライブラリの使用を終える際に呼び出してください。
 *  InitializeSystem() で NFP ライブラリを初期化した場合には、この API でライブラリを終了します。
 */
void FinalizeSystem() NN_NOEXCEPT;

/**
 * @brief         タグをフォーマットします。
 *
 * @param[in]     deviceHandle      NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               フォーマットに成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRetry,             フォーマットに失敗しました。再試行によって成功する可能性があります。再試行にて 3 回失敗した場合は、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           フォーマットに失敗しました。フォーマット対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNotSupported,          NFP タグではありません。使用しているタグを確認する必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details       タグに書かれたデータが破損しており、かつバックアップが存在しない場合、フォーマットが必要になります。
 *                API ではタグをフォーマットした後に、乱数をタグのアプリケーション専用領域に書き込みます。@n
 *                タグが検知済みでマウントも喪失もされていない必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *                フォーマットを行うと、同時に @ref Flush が実行されまするためこの API の実行には 1 ～ 2 秒程度かかります。
 */
nn::Result Format(const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief  タグの管理情報を、マウント時に用意した内部バッファから取得します。
 *
 * @param[out] pOutAdminInfo       管理情報の取得先です。
 * @param[in]  deviceHandle        NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               取得に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           取得に失敗しました。取得対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - @a pOutAdminInfo はアクセス可能なアドレスを指している必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details
 *  タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *  アプリケーション専用領域が作成されていない場合、
 *  @ref AdminInfo::applicationId, @ref AdminInfo::accessId には以下の値が格納されます。
 *  @li applicationId = 0
 *  @li accessId = 0
 */
nn::Result GetAdminInfo(AdminInfo* pOutAdminInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief   NFP タグのシステム領域の登録情報を、マウント時に用意した内部バッファから取得します。
 *
 * @param[out] pOutRegisterInfo        @ref RegisterInfo の取得先へのポインタ
 * @param[in]  deviceHandle            NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               取得に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           取得に失敗しました。取得対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRegister,          登録情報が設定されていません。amiibo 設定での設定が必要です。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - @a pOutRegisterInfo はアクセス可能なアドレスを指している必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details
 *  タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *  タグのニックネームには、アプリケーションが対応していない文字が含まれている場合があります。
 *  その場合でも、画面表示が崩れたり、アプリケーションの進行に支障が出ないように、対処してください。
 */
nn::Result GetRegisterInfo(RegisterInfoPrivate* pOutRegisterInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief  NFP タグのシステム領域の登録情報を、マウント時に用意した内部バッファに設定します。
 *
 * @param[in]  deviceHandle        NFC デバイスのハンドル
 * @param[in]  regInfo             初期登録情報です。
 *
 * @retresult
 *     @handleresult{ResultSuccess,               設定に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           設定に失敗しました。設定対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details
 *  タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *  登録情報の設定後、実際にタグに書き込むには @ref Flush を呼ぶ必要があります。@n
 *  この API において、 @a regInfo.registerDate は参照されません。実際に登録情報を初めて書き込みした日が登録日としてタグに書き込まれます。
 */
nn::Result SetRegisterInfo(const DeviceHandle& deviceHandle, const RegisterInfoPrivate& regInfo) NN_NOEXCEPT;

/**
 * @brief  NFP タグの登録情報を削除します。
 *
 * @param[in]  deviceHandle  NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               削除に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRetry,             削除に失敗しました。再試行によって成功する可能性があります。再試行にて 3 回失敗した場合は、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           削除に失敗しました。削除対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRegister,          登録情報が設定されていません。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @post
 *  - タグの @ref AdminInfo::registerInfo の @ref RegisterInfoFlags_CommonArea ビットがオフになり、登録情報が存在しない状態になります。
 *
 * @details
 *  タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *  登録情報を削除すると、同時に @ref Flush が実行されます。
 */
nn::Result DeleteRegisterInfo(const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief  NFP タグのアプリケーション専用領域の情報を削除します。
 *
 * @param[in]  deviceHandle  NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               削除に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRetry,             削除に失敗しました。再試行によって成功する可能性があります。再試行にて 3 回失敗した場合は、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           削除に失敗しました。削除対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedCreate,            アプリケーション専用領域が作成されていません。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @post
 *  - タグの @ref AdminInfo::registerInfo の @ref RegisterInfoFlags_ApplicationArea ビットがオフになり、アプリケーション専用領域が存在しない状態になります。
 *
 * @details
 *  タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *  アプリケーション専用領域を削除すると、同時に @ref Flush が実行されます。
 */
nn::Result DeleteApplicationArea(const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief  アプリケーション専用領域の有無を取得します。
 *
 * @param[out] outValue      アプリケーション専用領域の有無
 * @param[in]  deviceHandle  NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               取得に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           取得に失敗しました。取得対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details
 *  タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)
 */
nn::Result ExistsApplicationArea(bool *outValue, const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief   ニックネームとオーナーの設定を行う amiibo 設定を開始します。
 *
 * @param[out] pOutTagInfo      更新されたタグを示す情報を格納するバッファへのポインタを指定します。
 * @param[out] pOutDeviceHandle タグを更新した NFC デバイスのハンドルを格納するバッファへのポインタを指定します。
 * @param[out] pOutIsRegistered 更新後の登録情報の有無を格納するバッファへのポインタを指定します。
 * @param[out] pOutRegisterInfo 更新後の登録情報を格納するバッファへのポインタを指定します。@a pOutIsRegistered にて登録情報が「有り」を示す場合のみ参照してください。
 * @param[in]  startParam       amiibo 設定開始時必要な共通パラメータです。
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *  - @a pOutTagInfo はアクセス可能なアドレスを指している必要があります。
 *  - @a pOutDeviceHandle はアクセス可能なアドレスを指している必要があります。
 *  - @a pOutIsRegistered はアクセス可能なアドレスを指している必要があります。
 *  - @a pOutRegisterInfo はアクセス可能なアドレスを指している必要があります。
 *
 * @retresult
 *     @handleresult{ResultSuccess,           タグの更新に成功しました。}
 *     @handleresult{ResultNotUpdated,        タグは更新されていません。}
 * @endretresult
 *
 * @details
 *  ニックネームとオーナーの設定を行う amiibo 設定を開始します。本 API は、amiibo 設定が完了するまで戻りません。@n
 *  @ref ResultNotUpdated が返ってきた場合、@a pOutTagInfo 、@a pOutDeviceHandle 、@a pOutIsRegistered および @a pOutRegisterInfo には有効な情報が格納されていませんので、参照しないでください。@n
 *  ニックネームとオーナーの設定を行う amiibo 設定を開始する API は一般公開する API としても用意していますが、@n
 *  一般公開している API と違い、本 API では @ref TagInfo や @ref RegisterInfo を入力引数として指定しません。@n
 *  @ref TagInfo や @ref RegisterInfo を入力引数として指定しない本 API を使用するのは、本体設定のみであるため、Private API として用意しています。
 *
 */
nn::Result StartNicknameAndOwnerSettings(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, RegisterInfo* pOutRegisterInfo, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT;

/**
 * @brief   ゲームデータの消去を行う amiibo 設定を開始します。
 *
 * @param[out] pOutTagInfo      更新されたタグを示す情報を格納するバッファへのポインタを指定します。
 * @param[out] pOutDeviceHandle タグを更新した NFC デバイスのハンドルを格納するバッファへのポインタを指定します。
 * @param[in]  startParam       amiibo 設定開始時必要な共通パラメータです。
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *  - @a pOutTagInfo はアクセス可能なアドレスを指している必要があります。
 *  - @a pOutDeviceHandle はアクセス可能なアドレスを指している必要があります。
 *
 * @retresult
 *     @handleresult{ResultSuccess,           タグの更新に成功しました。}
 *     @handleresult{ResultNotUpdated,        タグは更新されていません。}
 * @endretresult
 *
 * @details
 *  ゲームデータの消去を行う amiibo 設定を開始します。本 API は、amiibo 設定が完了するまで戻りません。@n
 *  @ref ResultNotUpdated が返ってきた場合、@a pOutTagInfo および @a pOutDeviceHandle には有効な情報が格納されていませんので、参照しないでください。@n
 *  ゲームデータの消去を行う amiibo 設定を開始する API は一般公開する API としても用意していますが、@n
 *  一般公開している API と違い、本 API では @ref TagInfo を入力引数として指定しません。@n
 *  @ref TagInfo を入力引数として指定しない本 API を使用するのは、本体設定のみであるため、Private API として用意しています。
 *
 */
nn::Result StartGameDataEraser(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT;

/**
 * @brief   データの復旧を行う amiibo 設定を開始します。
 *
 * @param[out] pOutTagInfo      更新されたタグを示す情報を格納するバッファへのポインタを指定します。
 * @param[out] pOutDeviceHandle タグを更新した NFC デバイスのハンドルを格納するバッファへのポインタを指定します。
 * @param[in]  startParam       amiibo 設定開始時必要な共通パラメータです。
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *  - @a pOutTagInfo はアクセス可能なアドレスを指している必要があります。
 *  - @a pOutDeviceHandle はアクセス可能なアドレスを指している必要があります。
 *
 * @retresult
 *     @handleresult{ResultSuccess,           タグの更新に成功しました。}
 *     @handleresult{ResultNotUpdated,        タグは更新されていません。}
 * @endretresult
 *
 * @details
 *  データの復旧を行う amiibo 設定を開始します。本 API は、amiibo 設定が完了するまで戻りません。@n
 *  @ref ResultNotUpdated が返ってきた場合、@a pOutTagInfo および @a pOutDeviceHandle には有効な情報が格納されていませんので、参照しないでください。@n
 *  データの復旧を行う amiibo 設定を開始する API は一般公開する API としても用意していますが、@n
 *  一般公開している API と違い、本 API では @ref TagInfo を入力引数として指定しません。@n
 *  @ref TagInfo を入力引数として指定しない本 API を使用するのは、本体設定のみであるため、Private API として用意しています。
 *
 */
nn::Result StartRestorer(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT;

/**
 * @brief   フォーマットを行う amiibo 設定を開始します。
 *
 * @param[out] pOutTagInfo      更新されたタグを示す情報を格納するバッファへのポインタを指定します。
 * @param[out] pOutDeviceHandle タグを更新した NFC デバイスのハンドルを格納するバッファへのポインタを指定します。
 * @param[in]  startParam       amiibo 設定開始時必要な共通パラメータです。
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *  - @a pOutTagInfo はアクセス可能なアドレスを指している必要があります。
 *  - @a pOutDeviceHandle はアクセス可能なアドレスを指している必要があります。
 *
 * @retresult
 *     @handleresult{ResultSuccess,           タグの更新に成功しました。}
 *     @handleresult{ResultNotUpdated,        タグは更新されていません。}
 * @endretresult
 *
 * @details
 *  フォーマットを行う amiibo 設定を開始します。本 API は、amiibo 設定が完了するまで戻りません。@n
 *  @ref ResultNotUpdated が返ってきた場合、@a pOutTagInfo および @a pOutDeviceHandle には有効な情報が格納されていませんので、参照しないでください。@n
 *  amiibo 設定を開始する他の API とは異なり、フォーマットについては、本体設定でのみ利用されるため、一般公開 API は用意されていません。
 *
 */
nn::Result StartFormatter(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT;

/**
 * @}
 */

}  // nfp
}  // nn
